<?php
// Database configuration
define('DB_PATH', __DIR__ . '/hospital_system.db');

// Multi-tenant configuration
define('TENANT_COOKIE', 'hospital_tenant');
define('SESSION_TIMEOUT', 3600); // 1 hour

// User roles
define('ROLES', [
    'super_admin' => 'Super Administrator',
    'hospital_admin' => 'Hospital Administrator',
    'doctor' => 'Doctor',
    'nurse' => 'Nurse',
    'lab_scientist' => 'Lab Scientist',
    'lab_technician' => 'Lab Technician',
    'pharmacist' => 'Pharmacist',
    'pharmacy_technician' => 'Pharmacy Technician',
    'accountant' => 'Accountant',
    'record_manager' => 'Record Manager',
    'receptionist' => 'Receptionist',
    'cook' => 'Cook',
    'cleaner' => 'Cleaner',
    'admin_staff' => 'Administrative Staff'
]);

// Initialize session
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Database connection
class Database {
    private static $instance = null;
    private $db;

    private function __construct() {
        try {
            $this->db = new PDO('sqlite:' . DB_PATH);
            $this->db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->initializeTables();
        } catch (PDOException $e) {
            die("Database connection failed: " . $e->getMessage());
        }
    }

    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new Database();
        }
        return self::$instance;
    }

    public function getConnection() {
        return $this->db;
    }

    private function initializeTables() {
        $sql = "
        -- Tenants table (Hospitals)
        CREATE TABLE IF NOT EXISTS tenants (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            name TEXT NOT NULL,
            subdomain TEXT UNIQUE NOT NULL,
            address TEXT,
            phone TEXT,
            email TEXT,
            logo_url TEXT,
            registration_date DATE DEFAULT CURRENT_DATE,
            status TEXT DEFAULT 'pending',
            license_number TEXT,
            patient_card_fee DECIMAL(10,2) DEFAULT 5.00,
            email_notifications INTEGER DEFAULT 1,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP
        );

        -- Users table with extended roles
        CREATE TABLE IF NOT EXISTS users (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NULL,
            username TEXT NOT NULL,
            password TEXT NOT NULL,
            full_name TEXT NOT NULL,
            email TEXT,
            phone TEXT,
            role TEXT NOT NULL,
            department TEXT,
            employee_number TEXT,
            status TEXT DEFAULT 'active',
            created_by INTEGER,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (created_by) REFERENCES users(id),
            UNIQUE(username)
        );

        -- Patients table
        CREATE TABLE IF NOT EXISTS patients (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            patient_number TEXT NOT NULL,
            full_name TEXT NOT NULL,
            date_of_birth DATE,
            gender TEXT,
            blood_group TEXT,
            phone TEXT,
            email TEXT,
            address TEXT,
            emergency_contact TEXT,
            emergency_phone TEXT,
            status TEXT DEFAULT 'active',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            UNIQUE(tenant_id, patient_number)
        );

        -- Doctors table
        CREATE TABLE IF NOT EXISTS doctors (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            user_id INTEGER,
            doctor_number TEXT NOT NULL,
            full_name TEXT NOT NULL,
            specialization TEXT,
            qualification TEXT,
            phone TEXT,
            email TEXT,
            consultation_fee REAL,
            status TEXT DEFAULT 'active',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (user_id) REFERENCES users(id),
            UNIQUE(tenant_id, doctor_number)
        );

        -- Appointments table
        CREATE TABLE IF NOT EXISTS appointments (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            patient_id INTEGER NOT NULL,
            doctor_id INTEGER NOT NULL,
            appointment_date DATE NOT NULL,
            appointment_time TIME NOT NULL,
            reason TEXT,
            status TEXT DEFAULT 'scheduled',
            notes TEXT,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (patient_id) REFERENCES patients(id),
            FOREIGN KEY (doctor_id) REFERENCES doctors(id)
        );

        -- Billing table
        CREATE TABLE IF NOT EXISTS billing (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            patient_id INTEGER NOT NULL,
            appointment_id INTEGER,
            invoice_number TEXT NOT NULL,
            description TEXT,
            amount REAL NOT NULL,
            payment_status TEXT DEFAULT 'pending',
            payment_method TEXT,
            payment_date DATETIME,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (patient_id) REFERENCES patients(id),
            FOREIGN KEY (appointment_id) REFERENCES appointments(id),
            UNIQUE(tenant_id, invoice_number)
        );

        -- Medical records table (updated for prescriptions workflow)
        CREATE TABLE IF NOT EXISTS medical_records (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            patient_id INTEGER NOT NULL,
            doctor_id INTEGER,
            appointment_id INTEGER,
            consultation_type TEXT DEFAULT 'outpatient',
            chief_complaint TEXT,
            diagnosis TEXT,
            notes TEXT,
            recorded_by INTEGER,
            record_type TEXT DEFAULT 'consultation',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (patient_id) REFERENCES patients(id),
            FOREIGN KEY (doctor_id) REFERENCES doctors(id),
            FOREIGN KEY (appointment_id) REFERENCES appointments(id),
            FOREIGN KEY (recorded_by) REFERENCES users(id)
        );

        -- Prescriptions table (NEW)
        CREATE TABLE IF NOT EXISTS prescriptions (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            patient_id INTEGER NOT NULL,
            medical_record_id INTEGER NOT NULL,
            doctor_id INTEGER NOT NULL,
            prescription_number TEXT NOT NULL,
            status TEXT DEFAULT 'pending',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (patient_id) REFERENCES patients(id),
            FOREIGN KEY (medical_record_id) REFERENCES medical_records(id),
            FOREIGN KEY (doctor_id) REFERENCES users(id)
        );

        -- Prescription items table (drugs prescribed)
        CREATE TABLE IF NOT EXISTS prescription_items (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            prescription_id INTEGER NOT NULL,
            drug_id INTEGER,
            drug_name TEXT NOT NULL,
            dosage TEXT NOT NULL,
            frequency TEXT NOT NULL,
            duration TEXT NOT NULL,
            quantity INTEGER NOT NULL,
            instructions TEXT,
            status TEXT DEFAULT 'pending',
            invoice_id INTEGER,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (prescription_id) REFERENCES prescriptions(id),
            FOREIGN KEY (drug_id) REFERENCES pharmacy_inventory(id)
        );

        -- Lab test requests table (NEW)
        CREATE TABLE IF NOT EXISTS lab_requests (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            patient_id INTEGER NOT NULL,
            medical_record_id INTEGER,
            doctor_id INTEGER,
            request_number TEXT NOT NULL,
            status TEXT DEFAULT 'pending',
            priority TEXT DEFAULT 'routine',
            notes TEXT,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (patient_id) REFERENCES patients(id),
            FOREIGN KEY (medical_record_id) REFERENCES medical_records(id),
            FOREIGN KEY (doctor_id) REFERENCES users(id)
        );

        -- Lab request items table (tests requested)
        CREATE TABLE IF NOT EXISTS lab_request_items (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            lab_request_id INTEGER NOT NULL,
            test_id INTEGER,
            test_name TEXT NOT NULL,
            status TEXT DEFAULT 'pending',
            invoice_id INTEGER,
            result TEXT,
            reference_range TEXT,
            notes TEXT,
            performed_by INTEGER,
            result_date DATETIME,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (lab_request_id) REFERENCES lab_requests(id),
            FOREIGN KEY (test_id) REFERENCES lab_tests(id),
            FOREIGN KEY (performed_by) REFERENCES users(id)
        );

        -- Patient reports table (NEW) - for staff to write reports on patients
        CREATE TABLE IF NOT EXISTS patient_reports (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            patient_id INTEGER NOT NULL,
            report_number TEXT NOT NULL,
            report_type TEXT NOT NULL,
            report_title TEXT NOT NULL,
            report_content TEXT NOT NULL,
            written_by INTEGER NOT NULL,
            writer_role TEXT NOT NULL,
            writer_name TEXT NOT NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (patient_id) REFERENCES patients(id),
            FOREIGN KEY (written_by) REFERENCES users(id)
        );

        -- Nurse observations table (NEW) - for nurses to record vitals and observations
        CREATE TABLE IF NOT EXISTS nurse_observations (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            patient_id INTEGER NOT NULL,
            observation_number TEXT NOT NULL,
            blood_pressure TEXT,
            temperature REAL,
            weight REAL,
            pulse_rate INTEGER,
            respiratory_rate INTEGER,
            oxygen_saturation INTEGER,
            blood_sugar REAL,
            observations TEXT NOT NULL,
            nurse_id INTEGER NOT NULL,
            nurse_name TEXT NOT NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (patient_id) REFERENCES patients(id),
            FOREIGN KEY (nurse_id) REFERENCES users(id)
        );

        -- Patient Admissions table
        CREATE TABLE IF NOT EXISTS admissions (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            patient_id INTEGER NOT NULL,
            doctor_id INTEGER NOT NULL,
            admission_number TEXT NOT NULL,
            admission_date DATETIME NOT NULL,
            discharge_date DATETIME,
            room_number TEXT,
            bed_number TEXT,
            reason TEXT,
            diagnosis TEXT,
            status TEXT DEFAULT 'admitted',
            total_cost REAL DEFAULT 0,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (patient_id) REFERENCES patients(id),
            FOREIGN KEY (doctor_id) REFERENCES doctors(id),
            UNIQUE(tenant_id, admission_number)
        );

        -- Pharmacy Inventory table
        CREATE TABLE IF NOT EXISTS pharmacy_inventory (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            drug_name TEXT NOT NULL,
            generic_name TEXT,
            drug_code TEXT NOT NULL,
            category TEXT,
            dosage_form TEXT,
            strength TEXT,
            manufacturer TEXT,
            quantity INTEGER DEFAULT 0,
            reorder_level INTEGER DEFAULT 10,
            unit_price REAL,
            expiry_date DATE,
            status TEXT DEFAULT 'active',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            UNIQUE(tenant_id, drug_code)
        );

        -- Pharmacy Dispensing table
        CREATE TABLE IF NOT EXISTS pharmacy_dispensing (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            patient_id INTEGER NOT NULL,
            drug_id INTEGER NOT NULL,
            prescription_id INTEGER,
            quantity INTEGER NOT NULL,
            dispensed_by INTEGER NOT NULL,
            dispensed_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            unit_price REAL,
            total_price REAL,
            payment_status TEXT DEFAULT 'pending',
            notes TEXT,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (patient_id) REFERENCES patients(id),
            FOREIGN KEY (drug_id) REFERENCES pharmacy_inventory(id),
            FOREIGN KEY (dispensed_by) REFERENCES users(id)
        );

        -- Laboratory Tests table
        CREATE TABLE IF NOT EXISTS lab_tests (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            test_name TEXT NOT NULL,
            test_code TEXT NOT NULL,
            category TEXT,
            cost REAL,
            normal_range TEXT,
            description TEXT,
            status TEXT DEFAULT 'active',
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            UNIQUE(tenant_id, test_code)
        );

        -- Laboratory Test Orders table
        CREATE TABLE IF NOT EXISTS lab_orders (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            patient_id INTEGER NOT NULL,
            doctor_id INTEGER NOT NULL,
            test_id INTEGER NOT NULL,
            order_number TEXT NOT NULL,
            order_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            sample_collected_date DATETIME,
            result_date DATETIME,
            result TEXT,
            interpretation TEXT,
            status TEXT DEFAULT 'pending',
            tested_by INTEGER,
            cost REAL,
            payment_status TEXT DEFAULT 'pending',
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (patient_id) REFERENCES patients(id),
            FOREIGN KEY (doctor_id) REFERENCES doctors(id),
            FOREIGN KEY (test_id) REFERENCES lab_tests(id),
            FOREIGN KEY (tested_by) REFERENCES users(id),
            UNIQUE(tenant_id, order_number)
        );

        -- Restaurant/Catering Menu table
        CREATE TABLE IF NOT EXISTS restaurant_menu (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            item_name TEXT NOT NULL,
            item_code TEXT NOT NULL,
            category TEXT,
            description TEXT,
            unit_price REAL,
            availability TEXT DEFAULT 'available',
            dietary_info TEXT,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            UNIQUE(tenant_id, item_code)
        );

        -- Restaurant Orders table
        CREATE TABLE IF NOT EXISTS restaurant_orders (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            patient_id INTEGER,
            menu_item_id INTEGER NOT NULL,
            order_number TEXT NOT NULL,
            quantity INTEGER NOT NULL,
            unit_price REAL,
            total_price REAL,
            order_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            delivery_time TEXT,
            room_number TEXT,
            status TEXT DEFAULT 'pending',
            prepared_by INTEGER,
            delivered_by INTEGER,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (patient_id) REFERENCES patients(id),
            FOREIGN KEY (menu_item_id) REFERENCES restaurant_menu(id),
            FOREIGN KEY (prepared_by) REFERENCES users(id),
            FOREIGN KEY (delivered_by) REFERENCES users(id),
            UNIQUE(tenant_id, order_number)
        );

        -- Attendance Register table
        CREATE TABLE IF NOT EXISTS attendance (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            user_id INTEGER NOT NULL,
            attendance_date DATE NOT NULL,
            check_in_time DATETIME,
            check_out_time DATETIME,
            status TEXT DEFAULT 'present',
            notes TEXT,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (user_id) REFERENCES users(id),
            UNIQUE(tenant_id, user_id, attendance_date)
        );

        -- Accounting Transactions table
        CREATE TABLE IF NOT EXISTS accounting_transactions (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            tenant_id INTEGER NOT NULL,
            transaction_number TEXT NOT NULL,
            transaction_date DATE NOT NULL,
            transaction_type TEXT NOT NULL,
            category TEXT,
            description TEXT,
            amount REAL NOT NULL,
            payment_method TEXT,
            reference_type TEXT,
            reference_id INTEGER,
            created_by INTEGER,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (tenant_id) REFERENCES tenants(id),
            FOREIGN KEY (created_by) REFERENCES users(id),
            UNIQUE(tenant_id, transaction_number)
        );
        ";

        $this->db->exec($sql);
        $this->seedDefaultData();
    }

    private function seedDefaultData() {
        // Check if super admin exists
        $stmt = $this->db->query("SELECT COUNT(*) FROM users WHERE role = 'super_admin'");
        if ($stmt->fetchColumn() == 0) {
            // Create super admin (password: superadmin123)
            $hashedPassword = password_hash('superadmin123', PASSWORD_DEFAULT);
            $this->db->exec("
                INSERT INTO users (tenant_id, username, password, full_name, email, role)
                VALUES (NULL, 'superadmin', '$hashedPassword', 'Super Administrator', 'superadmin@system.com', 'super_admin')
            ");
        }

        // Check if demo tenant exists
        $stmt = $this->db->query("SELECT COUNT(*) FROM tenants WHERE subdomain = 'demo'");
        if ($stmt->fetchColumn() == 0) {
            // Create demo tenant
            $this->db->exec("
                INSERT INTO tenants (name, subdomain, address, phone, email, status)
                VALUES ('City General Hospital', 'demo', '123 Medical Center Drive', '+1-555-0100', 'info@citygeneralhospital.com', 'active')
            ");

            $tenantId = $this->db->lastInsertId();

            // Create hospital admin user (password: admin123)
            $hashedPassword = password_hash('admin123', PASSWORD_DEFAULT);
            $this->db->exec("
                INSERT INTO users (tenant_id, username, password, full_name, email, role, employee_number)
                VALUES ($tenantId, 'admin', '$hashedPassword', 'Hospital Administrator', 'admin@citygeneralhospital.com', 'hospital_admin', 'EMP001')
            ");

            // Create sample doctor
            $hashedPasswordDoc = password_hash('doctor123', PASSWORD_DEFAULT);
            $this->db->exec("
                INSERT INTO users (tenant_id, username, password, full_name, email, role, employee_number, department)
                VALUES ($tenantId, 'doctor1', '$hashedPasswordDoc', 'Dr. Sarah Johnson', 'sarah.j@citygeneralhospital.com', 'doctor', 'EMP002', 'Cardiology')
            ");

            $userId = $this->db->lastInsertId();

            $this->db->exec("
                INSERT INTO doctors (tenant_id, user_id, doctor_number, full_name, specialization, qualification, phone, email, consultation_fee)
                VALUES ($tenantId, $userId, 'DOC001', 'Dr. Sarah Johnson', 'Cardiology', 'MD, FACC', '+1-555-0101', 'sarah.j@citygeneralhospital.com', 150.00)
            ");

            // Create sample pharmacist
            $hashedPasswordPharm = password_hash('pharm123', PASSWORD_DEFAULT);
            $this->db->exec("
                INSERT INTO users (tenant_id, username, password, full_name, email, role, employee_number, department)
                VALUES ($tenantId, 'pharmacist1', '$hashedPasswordPharm', 'John Pharmacist', 'john.p@citygeneralhospital.com', 'pharmacist', 'EMP003', 'Pharmacy')
            ");

            // Create demo users for all 14 roles
            $demoUsers = [
                ['nurse', 'nurse123', 'Emily Nurse', 'emily.n@citygeneralhospital.com', 'nurse', 'EMP004', 'Nursing'],
                ['lab_scientist', 'lab123', 'Michael Lab', 'michael.l@citygeneralhospital.com', 'lab_scientist', 'EMP005', 'Laboratory'],
                ['lab_technician', 'labtech123', 'Sarah LabTech', 'sarah.lt@citygeneralhospital.com', 'lab_technician', 'EMP006', 'Laboratory'],
                ['pharmacy_technician', 'pharmtech123', 'David PharmTech', 'david.pt@citygeneralhospital.com', 'pharmacy_technician', 'EMP007', 'Pharmacy'],
                ['accountant', 'account123', 'Lisa Accountant', 'lisa.a@citygeneralhospital.com', 'accountant', 'EMP008', 'Finance'],
                ['record_manager', 'record123', 'James Recorder', 'james.r@citygeneralhospital.com', 'record_manager', 'EMP009', 'Records'],
                ['receptionist', 'reception123', 'Anna Reception', 'anna.r@citygeneralhospital.com', 'receptionist', 'EMP010', 'Front Desk'],
                ['cook', 'cook123', 'Carlos Cook', 'carlos.c@citygeneralhospital.com', 'cook', 'EMP011', 'Kitchen'],
                ['cleaner', 'clean123', 'Maria Cleaner', 'maria.c@citygeneralhospital.com', 'cleaner', 'EMP012', 'Housekeeping'],
                ['admin_staff', 'adminstaff123', 'Robert Admin', 'robert.as@citygeneralhospital.com', 'admin_staff', 'EMP013', 'Administration']
            ];

            foreach ($demoUsers as $user) {
                $hashedPassword = password_hash($user[1], PASSWORD_DEFAULT);
                $this->db->exec("
                    INSERT INTO users (tenant_id, username, password, full_name, email, role, employee_number, department)
                    VALUES ($tenantId, '{$user[0]}', '$hashedPassword', '{$user[2]}', '{$user[3]}', '{$user[4]}', '{$user[5]}', '{$user[6]}')
                ");
            }

            // Create sample patient
            $this->db->exec("
                INSERT INTO patients (tenant_id, patient_number, full_name, date_of_birth, gender, blood_group, phone, email, address, emergency_contact, emergency_phone)
                VALUES ($tenantId, 'PAT001', 'John Smith', '1985-03-15', 'Male', 'O+', '+1-555-1001', 'john.smith@email.com', '456 Oak Street', 'Jane Smith', '+1-555-1002')
            ");

            // Create sample lab tests
            $this->db->exec("
                INSERT INTO lab_tests (tenant_id, test_name, test_code, category, cost, normal_range)
                VALUES
                    ($tenantId, 'Complete Blood Count', 'CBC001', 'Hematology', 50.00, 'WBC: 4-11, RBC: 4.5-5.5'),
                    ($tenantId, 'Blood Glucose', 'GLU001', 'Chemistry', 30.00, '70-100 mg/dL'),
                    ($tenantId, 'Lipid Profile', 'LIP001', 'Chemistry', 75.00, 'Total Cholesterol < 200 mg/dL')
            ");

            // Create sample pharmacy items
            $this->db->exec("
                INSERT INTO pharmacy_inventory (tenant_id, drug_name, generic_name, drug_code, category, dosage_form, strength, quantity, unit_price, expiry_date)
                VALUES
                    ($tenantId, 'Paracetamol', 'Acetaminophen', 'DRG001', 'Analgesic', 'Tablet', '500mg', 500, 0.50, '2026-12-31'),
                    ($tenantId, 'Amoxicillin', 'Amoxicillin', 'DRG002', 'Antibiotic', 'Capsule', '250mg', 300, 1.20, '2025-08-30'),
                    ($tenantId, 'Ibuprofen', 'Ibuprofen', 'DRG003', 'NSAID', 'Tablet', '400mg', 400, 0.75, '2026-06-15')
            ");

            // Create sample menu items
            $this->db->exec("
                INSERT INTO restaurant_menu (tenant_id, item_name, item_code, category, description, unit_price, dietary_info)
                VALUES
                    ($tenantId, 'Grilled Chicken Breast', 'MEAL001', 'Main Course', 'Healthy grilled chicken with vegetables', 12.50, 'High Protein, Low Fat'),
                    ($tenantId, 'Vegetable Soup', 'MEAL002', 'Soup', 'Fresh vegetable soup', 5.00, 'Vegetarian, Low Calorie'),
                    ($tenantId, 'Fruit Salad', 'MEAL003', 'Dessert', 'Fresh seasonal fruits', 4.50, 'Vegan, Low Fat')
            ");
        }
    }
}

// Tenant management
class TenantManager {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    public function getCurrentTenant() {
        // Super admin has no tenant
        if (isset($_SESSION['role']) && $_SESSION['role'] === 'super_admin') {
            return null;
        }

        // Check session first
        if (isset($_SESSION['tenant_id'])) {
            return $this->getTenantById($_SESSION['tenant_id']);
        }

        // Check cookie
        if (isset($_COOKIE[TENANT_COOKIE])) {
            $tenant = $this->getTenantBySubdomain($_COOKIE[TENANT_COOKIE]);
            if ($tenant) {
                $_SESSION['tenant_id'] = $tenant['id'];
                return $tenant;
            }
        }

        // Default to demo tenant
        return $this->getTenantBySubdomain('demo');
    }

    public function setCurrentTenant($subdomain) {
        $tenant = $this->getTenantBySubdomain($subdomain);
        if ($tenant) {
            $_SESSION['tenant_id'] = $tenant['id'];
            setcookie(TENANT_COOKIE, $subdomain, time() + (86400 * 30), '/');
            return true;
        }
        return false;
    }

    public function getTenantById($id) {
        $stmt = $this->db->prepare("SELECT * FROM tenants WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function getTenantBySubdomain($subdomain) {
        $stmt = $this->db->prepare("SELECT * FROM tenants WHERE subdomain = ?");
        $stmt->execute([$subdomain]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function getAllTenants() {
        $stmt = $this->db->query("SELECT * FROM tenants ORDER BY name");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getActiveTenants() {
        $stmt = $this->db->query("SELECT * FROM tenants WHERE status = 'active' ORDER BY name");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}

// Authentication
class Auth {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }

    public function login($username, $password, $tenantId = null) {
        // Super admin login (no tenant)
        if ($tenantId === null || $tenantId === 'super') {
            $stmt = $this->db->prepare("SELECT * FROM users WHERE username = ? AND role = 'super_admin' AND status = 'active'");
            $stmt->execute([$username]);
        } else {
            // Regular user login (with tenant)
            $stmt = $this->db->prepare("SELECT * FROM users WHERE username = ? AND tenant_id = ? AND status = 'active'");
            $stmt->execute([$username, $tenantId]);
        }

        $user = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($user && password_verify($password, $user['password'])) {
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['full_name'] = $user['full_name'];
            $_SESSION['role'] = $user['role'];
            $_SESSION['tenant_id'] = $user['tenant_id'];
            $_SESSION['last_activity'] = time();
            return true;
        }
        return false;
    }

    public function logout() {
        session_destroy();
        return true;
    }

    public function isLoggedIn() {
        if (isset($_SESSION['user_id']) && isset($_SESSION['last_activity'])) {
            if (time() - $_SESSION['last_activity'] > SESSION_TIMEOUT) {
                $this->logout();
                return false;
            }
            $_SESSION['last_activity'] = time();
            return true;
        }
        return false;
    }

    public function hasRole($roles) {
        if (!is_array($roles)) {
            $roles = [$roles];
        }
        return isset($_SESSION['role']) && in_array($_SESSION['role'], $roles);
    }

    public function isSuperAdmin() {
        return isset($_SESSION['role']) && $_SESSION['role'] === 'super_admin';
    }

    public function isHospitalAdmin() {
        return isset($_SESSION['role']) && $_SESSION['role'] === 'hospital_admin';
    }

    public function canManageUsers() {
        return $this->isSuperAdmin() || $this->isHospitalAdmin();
    }
}

// Helper function to escape output
function e($str) {
    return htmlspecialchars($str ?? '', ENT_QUOTES, 'UTF-8');
}

// Helper function to format currency
function formatCurrency($amount) {
    return '$' . number_format($amount, 2);
}

// Helper function to send email
function sendEmail($to, $subject, $message, $fromName = 'Hospital Management System', $fromEmail = 'noreply@hospital.com') {
    $headers = "From: $fromName <$fromEmail>\r\n";
    $headers .= "Reply-To: $fromEmail\r\n";
    $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
    $headers .= "MIME-Version: 1.0\r\n";

    return mail($to, $subject, $message, $headers);
}

// Helper function to get role display name
function getRoleName($role) {
    return ROLES[$role] ?? ucfirst(str_replace('_', ' ', $role));
}
