<?php
require_once 'config.php';

$auth = new Auth();
$tenantManager = new TenantManager();
$currentTenant = $tenantManager->getCurrentTenant();

// Handle actions
$action = $_GET['action'] ?? 'home';
$message = '';
$error = '';

// Handle login
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login'])) {
    $username = $_POST['username'] ?? '';
    $password = $_POST['password'] ?? '';
    $tenantId = $_POST['tenant_id'] ?? ($currentTenant['id'] ?? null);

    if ($auth->login($username, $password, $tenantId)) {
        if ($auth->isSuperAdmin()) {
            header('Location: ?action=super_admin_dashboard');
        } else {
            header('Location: ?action=dashboard');
        }
        exit;
    } else {
        $error = 'Invalid username or password';
    }
}

// Handle logout
if ($action === 'logout') {
    $auth->logout();
    header('Location: ?action=home');
    exit;
}

// Handle tenant switching
if (isset($_GET['switch_tenant'])) {
    $tenantManager->setCurrentTenant($_GET['switch_tenant']);
    header('Location: ?action=home');
    exit;
}

$pageTitle = ($auth->isSuperAdmin() ? 'Super Admin' : ($currentTenant['name'] ?? 'Hospital Management System')) . ' - Hospital Management System';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($pageTitle) ?></title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary-color: #2563eb;
            --secondary-color: #1e40af;
            --success-color: #10b981;
            --danger-color: #ef4444;
            --warning-color: #f59e0b;
            --dark-color: #1f2937;
            --light-color: #f3f4f6;
            --white-color: #ffffff;
            --text-dark: #111827;
            --text-light: #6b7280;
            --border-color: #e5e7eb;
            --shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
            --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
            line-height: 1.6;
            color: var(--text-dark);
            background-color: var(--light-color);
        }

        /* Header */
        .header {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
            color: var(--white-color);
            padding: 1rem 0;
            box-shadow: var(--shadow-lg);
            position: sticky;
            top: 0;
            z-index: 1000;
        }

        .header-content {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
            flex-wrap: wrap;
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            font-size: 1.5rem;
            font-weight: 700;
        }

        .logo-icon {
            width: 40px;
            height: 40px;
            background: var(--white-color);
            color: var(--primary-color);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: bold;
        }

        .nav {
            display: flex;
            gap: 1rem;
            align-items: center;
            flex-wrap: wrap;
        }

        .nav a, .nav button {
            color: var(--white-color);
            text-decoration: none;
            padding: 0.5rem 1rem;
            border-radius: 6px;
            transition: background-color 0.3s;
            border: none;
            background: transparent;
            cursor: pointer;
            font-size: 1rem;
        }

        .nav a:hover, .nav button:hover {
            background-color: rgba(255, 255, 255, 0.1);
        }

        /* Container */
        .container {
            max-width: 1200px;
            margin: 2rem auto;
            padding: 0 1rem;
        }

        /* Hero Section */
        .hero {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: var(--white-color);
            padding: 4rem 2rem;
            border-radius: 12px;
            margin-bottom: 2rem;
            box-shadow: var(--shadow-lg);
            text-align: center;
        }

        .hero h1 {
            font-size: 2.5rem;
            margin-bottom: 1rem;
            font-weight: 700;
        }

        .hero p {
            font-size: 1.25rem;
            margin-bottom: 2rem;
            opacity: 0.95;
        }

        .hero .tenant-info {
            background: rgba(255, 255, 255, 0.1);
            padding: 1rem;
            border-radius: 8px;
            display: inline-block;
            margin-top: 1rem;
        }

        /* Cards */
        .card {
            background: var(--white-color);
            border-radius: 12px;
            padding: 1.5rem;
            box-shadow: var(--shadow);
            margin-bottom: 1.5rem;
        }

        .card h2 {
            font-size: 1.5rem;
            margin-bottom: 1rem;
            color: var(--text-dark);
        }

        .card h3 {
            font-size: 1.25rem;
            margin-bottom: 0.75rem;
            color: var(--text-dark);
        }

        /* Grid */
        .grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        /* Feature Cards */
        .feature-card {
            background: var(--white-color);
            border-radius: 12px;
            padding: 2rem;
            text-align: center;
            box-shadow: var(--shadow);
            transition: transform 0.3s, box-shadow 0.3s;
            cursor: pointer;
        }

        .feature-card:hover {
            transform: translateY(-5px);
            box-shadow: var(--shadow-lg);
        }

        .feature-icon {
            width: 60px;
            height: 60px;
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: var(--white-color);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 1rem;
            font-size: 1.5rem;
            font-weight: bold;
        }

        .feature-card h3 {
            font-size: 1.25rem;
            margin-bottom: 0.5rem;
        }

        .feature-card p {
            color: var(--text-light);
            font-size: 0.95rem;
        }

        /* Forms */
        .form-group {
            margin-bottom: 1.25rem;
        }

        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 500;
            color: var(--text-dark);
        }

        .form-control {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid var(--border-color);
            border-radius: 6px;
            font-size: 1rem;
            transition: border-color 0.3s, box-shadow 0.3s;
        }

        .form-control:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }

        /* Buttons */
        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 6px;
            font-size: 1rem;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s;
            display: inline-block;
            text-decoration: none;
            text-align: center;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: var(--white-color);
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
        }

        .btn-success {
            background-color: var(--success-color);
            color: var(--white-color);
        }

        .btn-danger {
            background-color: var(--danger-color);
            color: var(--white-color);
        }

        .btn-secondary {
            background-color: var(--text-light);
            color: var(--white-color);
        }

        .btn-block {
            width: 100%;
        }

        /* Alerts */
        .alert {
            padding: 1rem;
            border-radius: 6px;
            margin-bottom: 1rem;
        }

        .alert-success {
            background-color: #d1fae5;
            color: #065f46;
            border: 1px solid #6ee7b7;
        }

        .alert-error {
            background-color: #fee2e2;
            color: #991b1b;
            border: 1px solid #fca5a5;
        }

        /* Table */
        .table {
            width: 100%;
            border-collapse: collapse;
            background: var(--white-color);
            border-radius: 8px;
            overflow: hidden;
        }

        .table th {
            background-color: var(--primary-color);
            color: var(--white-color);
            padding: 1rem;
            text-align: left;
            font-weight: 600;
        }

        .table td {
            padding: 1rem;
            border-bottom: 1px solid var(--border-color);
        }

        .table tr:hover {
            background-color: var(--light-color);
        }

        /* Stats */
        .stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: var(--white-color);
            padding: 1.5rem;
            border-radius: 12px;
            box-shadow: var(--shadow);
            border-left: 4px solid var(--primary-color);
        }

        .stat-card.success {
            border-left-color: var(--success-color);
        }

        .stat-card.warning {
            border-left-color: var(--warning-color);
        }

        .stat-card.danger {
            border-left-color: var(--danger-color);
        }

        .stat-value {
            font-size: 2rem;
            font-weight: 700;
            color: var(--text-dark);
        }

        .stat-label {
            color: var(--text-light);
            font-size: 0.9rem;
            margin-top: 0.25rem;
        }

        /* Footer */
        .footer {
            background-color: var(--dark-color);
            color: var(--white-color);
            padding: 2rem 0;
            margin-top: 4rem;
            text-align: center;
        }

        .footer-content {
            max-width: 1200px;
            margin: 0 auto;
            padding: 0 1rem;
        }

        /* Tenant Selector */
        .tenant-selector {
            background: var(--white-color);
            padding: 0.5rem;
            border-radius: 6px;
            display: inline-block;
        }

        .tenant-selector select {
            padding: 0.5rem;
            border: 1px solid var(--border-color);
            border-radius: 4px;
            font-size: 0.9rem;
        }

        /* Responsive */
        @media (max-width: 768px) {
            .hero h1 {
                font-size: 1.75rem;
            }

            .hero p {
                font-size: 1rem;
            }

            .header-content {
                flex-direction: column;
                gap: 1rem;
            }

            .nav {
                width: 100%;
                justify-content: center;
            }

            .grid {
                grid-template-columns: 1fr;
            }

            .stats {
                grid-template-columns: 1fr;
            }

            .table {
                font-size: 0.875rem;
            }

            .table th, .table td {
                padding: 0.5rem;
            }
        }

        @media (max-width: 480px) {
            .container {
                padding: 0 0.5rem;
            }

            .hero {
                padding: 2rem 1rem;
            }

            .card {
                padding: 1rem;
            }

            .feature-card {
                padding: 1.5rem;
            }
        }
    </style>
</head>
<body>
    <!-- Header -->
    <header class="header">
        <div class="header-content">
            <div class="logo">
                <div class="logo-icon">H+</div>
                <span><?= $auth->isSuperAdmin() ? 'Super Admin Portal' : e($currentTenant['name'] ?? 'HMS') ?></span>
            </div>
            <nav class="nav">
                <?php if ($auth->isLoggedIn()): ?>
                    <?php if ($auth->isSuperAdmin()): ?>
                        <a href="?action=super_admin_dashboard">Dashboard</a>
                        <a href="?action=super_admin_hospitals">Hospitals</a>
                        <a href="?action=super_admin_users">All Users</a>
                    <?php else: ?>
                        <a href="?action=dashboard">Dashboard</a>
                        <?php if ($auth->canManageUsers()): ?>
                            <a href="?action=users">Users</a>
                        <?php endif; ?>
                        <a href="?action=patients">Patients</a>
                        <a href="?action=doctors">Doctors</a>
                        <a href="?action=consultation">Consultation</a>
                        <a href="?action=appointments">Appointments</a>
                        <a href="?action=billing">Billing</a>
                        <a href="?action=admissions">Admissions</a>
                        <a href="?action=pharmacy">Pharmacy</a>
                        <a href="?action=pharmacy_prescriptions">Rx Dispensing</a>
                        <a href="?action=laboratory">Lab</a>
                        <a href="?action=lab_requests">Lab Requests</a>
                        <a href="?action=restaurant">Restaurant</a>
                        <a href="?action=restaurant_orders">Meal Orders</a>
                        <a href="?action=accounting">Accounting</a>
                        <a href="?action=attendance">Attendance</a>
                    <?php endif; ?>
                    <?php if (in_array($_SESSION['role'], ['hospital_admin', 'nurse', 'doctor'])): ?>
                        <a href="?action=nurse_station">Nurse Station</a>
                    <?php endif; ?>
                    <?php if (in_array($_SESSION['role'], ['nurse', 'hospital_admin'])): ?>
                        <a href="?action=nurse_observations">Nurse Observations</a>
                    <?php endif; ?>
                    <?php if (in_array($_SESSION['role'], ['nurse', 'doctor', 'pharmacist', 'pharmacy_technician', 'lab_scientist', 'lab_technician'])): ?>
                        <a href="?action=patient_reports">Patient Reports</a>
                    <?php endif; ?>
                    <?php if ($_SESSION['role'] === 'hospital_admin'): ?>
                        <a href="?action=settings">Settings</a>
                    <?php endif; ?>
                    <a href="?action=profile">My Profile</a>
                    <span style="opacity: 0.8;">Welcome, <?= e($_SESSION['full_name']) ?></span>
                    <a href="?action=logout">Logout</a>
                <?php else: ?>
                    <a href="?action=home">Home</a>
                    <a href="?action=register_hospital">Register Hospital</a>
                    <a href="?action=login">Login</a>
                    <a href="?action=super_login">Super Admin</a>
                <?php endif; ?>
            </nav>
        </div>
    </header>

    <!-- Main Content -->
    <main class="container">
        <?php if ($error): ?>
            <div class="alert alert-error"><?= e($error) ?></div>
        <?php endif; ?>

        <?php if ($message): ?>
            <div class="alert alert-success"><?= e($message) ?></div>
        <?php endif; ?>

        <?php
        // Route to appropriate page
        if ($auth->isLoggedIn()) {
            // Super Admin Routes
            if ($auth->isSuperAdmin()) {
                switch ($action) {
                    case 'super_admin_dashboard':
                        include 'pages/super_admin_dashboard.php';
                        break;
                    case 'super_admin_hospitals':
                        include 'pages/super_admin_hospitals.php';
                        break;
                    case 'super_admin_users':
                        include 'pages/super_admin_users.php';
                        break;
                    case 'super_admin_reports':
                        include 'pages/super_admin_reports.php';
                        break;
                    default:
                        include 'pages/super_admin_dashboard.php';
                }
            } else {
                // Regular User Routes
                switch ($action) {
                    case 'dashboard':
                        include 'pages/dashboard.php';
                        break;
                    case 'users':
                        include 'pages/users.php';
                        break;
                    case 'patients':
                        include 'pages/patients.php';
                        break;
                    case 'doctors':
                        include 'pages/doctors.php';
                        break;
                    case 'consultation':
                        include 'pages/consultation.php';
                        break;
                    case 'appointments':
                        include 'pages/appointments.php';
                        break;
                    case 'billing':
                        include 'pages/billing.php';
                        break;
                    case 'admissions':
                        include 'pages/admissions.php';
                        break;
                    case 'pharmacy':
                        include 'pages/pharmacy.php';
                        break;
                    case 'pharmacy_prescriptions':
                        include 'pages/pharmacy_prescriptions.php';
                        break;
                    case 'laboratory':
                        include 'pages/laboratory.php';
                        break;
                    case 'lab_requests':
                        include 'pages/lab_requests.php';
                        break;
                    case 'restaurant':
                        include 'pages/restaurant.php';
                        break;
                    case 'restaurant_orders':
                        include 'pages/restaurant_orders.php';
                        break;
                    case 'accounting':
                        include 'pages/accounting.php';
                        break;
                    case 'attendance':
                        include 'pages/attendance.php';
                        break;
                    case 'profile':
                        include 'pages/profile.php';
                        break;
                    case 'settings':
                        include 'pages/settings.php';
                        break;
                    case 'nurse_station':
                        include 'pages/nurse_station.php';
                        break;
                    case 'receipt':
                        include 'pages/receipt.php';
                        break;
                    case 'patient_reports':
                        include 'pages/patient_reports.php';
                        break;
                    case 'nurse_observations':
                        include 'pages/nurse_observations.php';
                        break;
                    default:
                        include 'pages/dashboard.php';
                }
            }
        } else {
            // Public Routes
            switch ($action) {
                case 'login':
                    include 'pages/login.php';
                    break;
                case 'staff_login':
                    include 'pages/staff_login.php';
                    break;
                case 'admin_login':
                    include 'pages/admin_login.php';
                    break;
                case 'super_login':
                    include 'pages/super_login.php';
                    break;
                case 'register_hospital':
                    include 'pages/register_hospital.php';
                    break;
                default:
                    include 'pages/home.php';
            }
        }
        ?>
    </main>

    <!-- Footer -->
    <footer class="footer">
        <div class="footer-content">
            <p>&copy; <?= date('Y') ?> <?= e($currentTenant['name']) ?>. All rights reserved.</p>
            <p style="margin-top: 0.5rem; opacity: 0.8;">
                Multi-Tenant Hospital Management System |
                <span class="tenant-selector">
                    Switch Hospital:
                    <select onchange="window.location.href='?switch_tenant=' + this.value">
                        <?php
                        $tenants = $tenantManager->getAllTenants();
                        foreach ($tenants as $tenant):
                        ?>
                            <option value="<?= e($tenant['subdomain']) ?>"
                                    <?= $tenant['id'] == $currentTenant['id'] ? 'selected' : '' ?>>
                                <?= e($tenant['name']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </span>
            </p>
        </div>
    </footer>
</body>
</html>
