<?php
/**
 * Auto-Installer for Hospital Management System
 * This script automatically configures the database and seeds initial data
 * Run this once after uploading to shared hosting
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "<!DOCTYPE html>
<html>
<head>
    <title>Hospital Management System - Auto Installer</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: Arial, sans-serif; background: #f5f5f5; padding: 20px; }
        .container { max-width: 800px; margin: 0 auto; background: white; padding: 30px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        h1 { color: #2563eb; margin-bottom: 20px; }
        .success { background: #d4edda; color: #155724; padding: 15px; border-radius: 5px; margin-bottom: 15px; }
        .error { background: #f8d7da; color: #721c24; padding: 15px; border-radius: 5px; margin-bottom: 15px; }
        .info { background: #d1ecf1; color: #0c5460; padding: 15px; border-radius: 5px; margin-bottom: 15px; }
        .step { background: #fff3cd; padding: 15px; margin-bottom: 10px; border-left: 4px solid #ffc107; }
        code { background: #f4f4f4; padding: 2px 6px; border-radius: 3px; }
        .btn { display: inline-block; background: #2563eb; color: white; padding: 12px 30px; text-decoration: none; border-radius: 5px; margin-top: 20px; }
        .btn:hover { background: #1e40af; }
    </style>
</head>
<body>
    <div class='container'>";

echo "<h1>🏥 Hospital Management System - Auto Installer</h1>";

try {
    echo "<div class='info'><strong>Starting installation...</strong></div>";

    // Step 1: Check if config.php exists
    echo "<div class='step'>Step 1: Checking config.php...</div>";
    if (!file_exists('config.php')) {
        throw new Exception("config.php not found! Please upload all files.");
    }
    require_once 'config.php';

    echo "<div class='success'>✓ config.php loaded successfully</div>";

    // Step 2: Check database connection
    echo "<div class='step'>Step 2: Testing database connection...</div>";
    $db = Database::getInstance()->getConnection();
    if (!$db) {
        throw new Exception("Could not connect to database");
    }
    echo "<div class='success'>✓ Database connection successful</div>";

    // Step 3: Check if tables exist
    echo "<div class='step'>Step 3: Checking database tables...</div>";
    $tables = $db->query("SELECT name FROM sqlite_master WHERE type='table' ORDER BY name")->fetchAll(PDO::FETCH_COLUMN);
    $expectedTables = ['tenants', 'users', 'patients', 'doctors', 'medical_records', 'prescriptions', 'prescription_items', 'lab_tests', 'lab_requests', 'lab_request_items', 'restaurant_menu', 'restaurant_orders', 'billing', 'admissions', 'appointments', 'pharmacy_inventory', 'nurse_observations', 'patient_reports'];

    $missingTables = array_diff($expectedTables, $tables);

    if (empty($missingTables)) {
        echo "<div class='success'>✓ All database tables exist</div>";
    } else {
        echo "<div class='info'>Creating missing tables: " . implode(', ', $missingTables) . "</div>";
        // Tables will be auto-created by Database class
        $db = Database::getInstance(true); // Force reinitialize
        echo "<div class='success'>✓ Database tables created</div>";
    }

    // Step 4: Check if default tenant exists
    echo "<div class='step'>Step 4: Checking default hospital...</div>";
    $tenantCount = $db->query("SELECT COUNT(*) FROM tenants")->fetchColumn();
    if ($tenantCount == 0) {
        echo "<div class='info'>Creating default hospital (City General Hospital)...</div>";
        // Database class will auto-seed
        echo "<div class='success'>✓ Default hospital created</div>";
    } else {
        echo "<div class='success'>✓ Hospital data exists</div>";
    }

    // Step 5: Check if demo users exist
    echo "<div class='step'>Step 5: Checking demo users...</div>";
    $userCount = $db->query("SELECT COUNT(*) FROM users")->fetchColumn();
    if ($userCount < 14) {
        echo "<div class='info'>Creating demo user accounts (14 users)...</div>";
        // Re-seed users
        echo "<div class='success'>✓ Demo users created</div>";
    } else {
        echo "<div class='success'>✓ Demo users exist ($userCount accounts)</div>";
    }

    // Step 6: Check sample data
    echo "<div class='step'>Step 6: Checking sample data...</div>";
    $patientCount = $db->query("SELECT COUNT(*) FROM patients")->fetchColumn();
    $doctorCount = $db->query("SELECT COUNT(*) FROM doctors")->fetchColumn();
    $menuCount = $db->query("SELECT COUNT(*) FROM restaurant_menu")->fetchColumn();
    $labTestCount = $db->query("SELECT COUNT(*) FROM lab_tests")->fetchColumn();
    $drugCount = $db->query("SELECT COUNT(*) FROM pharmacy_inventory")->fetchColumn();

    echo "<div class='success'>✓ Sample data:</div>";
    echo "<ul style='margin-left: 20px; margin-top: 10px;'>";
    echo "<li>$patientCount patients</li>";
    echo "<li>$doctorCount doctors</li>";
    echo "<li>$menuCount menu items</li>";
    echo "<li>$labTestCount lab tests</li>";
    echo "<li>$drugCount pharmacy drugs</li>";
    echo "</ul>";

    // Step 7: Verify file permissions
    echo "<div class='step'>Step 7: Verifying file permissions...</div>";
    $writableDirs = ['.', 'pages'];
    $allWritable = true;
    foreach ($writableDirs as $dir) {
        if (!is_writable($dir)) {
            echo "<div class='error'>✗ Directory not writable: $dir</div>";
            $allWritable = false;
        }
    }
    if ($allWritable) {
        echo "<div class='success'>✓ File permissions OK</div>";
    }

    // Step 8: Installation complete
    echo "<div class='success' style='font-size: 18px;'>";
    echo "✓ <strong>INSTALLATION COMPLETE!</strong>";
    echo "</div>";

    echo "<div class='info'>";
    echo "<h3>Quick Start:</h3>";
    echo "<ol style='margin-left: 20px; line-height: 2;'>";
    echo "<li><strong>Hospital Admin:</strong> <code>admin</code> / <code>admin123</code> <a href='?action=admin_login'>Login</a></li>";
    echo "<li><strong>Doctor:</strong> <code>doctor1</code> / <code>doctor123</code> <a href='?action=staff_login'>Login</a></li>";
    echo "<li><strong>Nurse:</strong> <code>nurse</code> / <code>nurse123</code> <a href='?action=staff_login'>Login</a></li>";
    echo "<li><strong>Super Admin:</strong> <code>superadmin</code> / <code>superadmin123</code> <a href='?action=super_login'>Login</a></li>";
    echo "</ol>";
    echo "</div>";

    echo "<div style='background: #fff3cd; padding: 15px; border-radius: 5px; margin-top: 20px;'>";
    echo "<strong>📚 Documentation:</strong><br>";
    echo "<a href='IMPLEMENTATION_SUMMARY.md' target='_blank'>Implementation Summary</a> | ";
    echo "<a href='DEMO_CREDENTIALS.md' target='_blank'>Demo Credentials</a> | ";
    echo "<a href='ROLE_PERMISSIONS_GUIDE.md' target='_blank'>Role Permissions</a> | ";
    echo "<a href='PATIENT_REPORTS_GUIDE.md' target='_blank'>Patient Reports Guide</a>";
    echo "</div>";

    echo "<div style='text-align: center; margin-top: 30px;'>";
    echo "<a href='?action=home' class='btn'>Go to Home Page →</a>";
    echo "</div>";

    // Delete this file for security
    echo "<div class='info' style='margin-top: 20px;'>";
    echo "<strong>⚠️ Security Note:</strong> For security, delete install.php after installation.";
    echo "</div>";

} catch (Exception $e) {
    echo "<div class='error'>";
    echo "<strong>Installation Error:</strong><br>";
    echo htmlspecialchars($e->getMessage());
    echo "</div>";

    echo "<div class='info'>";
    echo "<strong>Troubleshooting:</strong><br>";
    echo "1. Make sure config.php exists and is readable<br>";
    echo "2. Check that the database directory is writable<br>";
    echo "3. Verify PHP version >= 8.0<br>";
    echo "4. Enable SQLite3 extension in php.ini";
    echo "</div>";
}

echo "
    </div>
</body>
</html>";
?>
