<?php
$db = Database::getInstance()->getConnection();
$tenantId = $_SESSION['tenant_id'];
$userId = $_SESSION['user_id'];
$userRole = $_SESSION['role'];

// Check if user can access consultation
$canConsult = in_array($userRole, ['hospital_admin', 'doctor']);

if (!$canConsult) {
    header('Location: ?action=dashboard');
    exit;
}

$subAction = $_GET['sub'] ?? 'search';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['save_consultation'])) {
        // Create medical record
        $stmt = $db->prepare("
            INSERT INTO medical_records (tenant_id, patient_id, doctor_id, consultation_type, chief_complaint, diagnosis, notes, recorded_by, record_type)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'consultation')
        ");

        $stmt->execute([
            $tenantId,
            $_POST['patient_id'],
            $userId,
            'outpatient',
            $_POST['chief_complaint'],
            $_POST['diagnosis'],
            $_POST['notes'],
            $userId
        ]);

        $medicalRecordId = $db->lastInsertId();

        // Handle prescriptions if drugs were added
        if (isset($_POST['drugs']) && count($_POST['drugs']) > 0) {
            $prescriptionNumber = 'PRX' . date('Ymd') . str_pad($db->query("SELECT COUNT(*) FROM prescriptions WHERE tenant_id = $tenantId")->fetchColumn() + 1, 4, '0', STR_PAD_LEFT);

            $stmt = $db->prepare("
                INSERT INTO prescriptions (tenant_id, patient_id, medical_record_id, doctor_id, prescription_number)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([$tenantId, $_POST['patient_id'], $medicalRecordId, $userId, $prescriptionNumber]);
            $prescriptionId = $db->lastInsertId();

            // Add prescription items
            foreach ($_POST['drugs'] as $index => $drugId) {
                if (!empty($drugId) && !empty($_POST['drug_names'][$index])) {
                    $stmt = $db->prepare("
                        INSERT INTO prescription_items (tenant_id, prescription_id, drug_id, drug_name, dosage, frequency, duration, quantity, instructions)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([
                        $tenantId,
                        $prescriptionId,
                        $drugId,
                        $_POST['drug_names'][$index],
                        $_POST['dosages'][$index],
                        $_POST['frequencies'][$index],
                        $_POST['durations'][$index],
                        $_POST['quantities'][$index],
                        $_POST['instructions'][$index]
                    ]);
                }
            }
        }

        // Handle lab tests if tests were requested
        if (isset($_POST['lab_tests']) && count($_POST['lab_tests']) > 0) {
            $labRequestNumber = 'LR' . date('Ymd') . str_pad($db->query("SELECT COUNT(*) FROM lab_requests WHERE tenant_id = $tenantId")->fetchColumn() + 1, 4, '0', STR_PAD_LEFT);

            $stmt = $db->prepare("
                INSERT INTO lab_requests (tenant_id, patient_id, medical_record_id, doctor_id, request_number, notes)
                VALUES (?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $tenantId,
                $_POST['patient_id'],
                $medicalRecordId,
                $userId,
                $labRequestNumber,
                $_POST['lab_notes'] ?? ''
            ]);
            $labRequestId = $db->lastInsertId();

            // Add lab test items
            foreach ($_POST['lab_tests'] as $testId) {
                if (!empty($testId)) {
                    // Get test name
                    $testStmt = $db->prepare("SELECT test_name FROM lab_tests WHERE id = ?");
                    $testStmt->execute([$testId]);
                    $testName = $testStmt->fetchColumn();

                    $stmt = $db->prepare("
                        INSERT INTO lab_request_items (tenant_id, lab_request_id, test_id, test_name)
                        VALUES (?, ?, ?, ?)
                    ");
                    $stmt->execute([$tenantId, $labRequestId, $testId, $testName]);
                }
            }
        }

        $message = 'Consultation saved successfully!';
        $subAction = 'search';
    }
}

// Patient search
$searchResults = [];
$selectedPatient = null;

if (isset($_GET['search']) && !empty($_GET['search_query'])) {
    $searchQuery = '%' . $_GET['search_query'] . '%';
    $stmt = $db->prepare("
        SELECT * FROM patients
        WHERE tenant_id = ? AND (patient_number LIKE ? OR full_name LIKE ?)
        ORDER BY full_name
        LIMIT 20
    ");
    $stmt->execute([$tenantId, $searchQuery, $searchQuery]);
    $searchResults = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Get patient details if selected for consultation
if ($subAction === 'consult' && isset($_GET['patient_id'])) {
    $stmt = $db->prepare("SELECT * FROM patients WHERE id = ? AND tenant_id = ?");
    $stmt->execute([$_GET['patient_id'], $tenantId]);
    $selectedPatient = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($selectedPatient) {
        // Get available drugs from pharmacy
        $drugs = $db->query("SELECT * FROM pharmacy_inventory WHERE tenant_id = $tenantId ORDER BY drug_name")->fetchAll(PDO::FETCH_ASSOC);

        // Get available lab tests
        $labTests = $db->query("SELECT * FROM lab_tests WHERE tenant_id = $tenantId ORDER BY test_name")->fetchAll(PDO::FETCH_ASSOC);

        // Get patient medical history
        $historyStmt = $db->prepare("
            SELECT mr.*, u.full_name as doctor_name
            FROM medical_records mr
            LEFT JOIN users u ON mr.doctor_id = u.id
            WHERE mr.patient_id = ? AND mr.tenant_id = ? AND mr.record_type = 'consultation'
            ORDER BY mr.created_at DESC
            LIMIT 5
        ");
        $historyStmt->execute([$_GET['patient_id'], $tenantId]);
        $medicalHistory = $historyStmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
?>

<div class="hero">
    <h1>Doctor Consultation</h1>
    <p>Patient examination, prescription, and lab test requests</p>
</div>

<?php if (isset($message)): ?>
    <div class="alert alert-success"><?= e($message) ?></div>
<?php endif; ?>

<?php if ($subAction === 'search' || !$selectedPatient): ?>
    <!-- Patient Search -->
    <div class="card">
        <h2>Search Patient</h2>
        <p style="color: var(--text-dark); margin-bottom: 1.5rem;">
            Search by patient number or name to start consultation.
        </p>

        <form method="GET" action="?action=consultation">
            <input type="hidden" name="action" value="consultation">
            <input type="hidden" name="search" value="1">

            <div style="display: flex; gap: 1rem; flex-wrap: wrap;">
                <div style="flex: 1; min-width: 300px;">
                    <input type="text" name="search_query" class="form-control"
                           placeholder="Enter patient number (e.g., PAT00001) or patient name"
                           value="<?= e($_GET['search_query'] ?? '') ?>" required>
                </div>
                <button type="submit" class="btn btn-primary">Search Patient</button>
            </div>
        </form>
    </div>

    <?php if (count($searchResults) > 0): ?>
        <!-- Search Results -->
        <div class="card">
            <h2>Search Results (<?= count($searchResults) ?>)</h2>
            <div style="overflow-x: auto;">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Patient #</th>
                            <th>Full Name</th>
                            <th>DOB</th>
                            <th>Gender</th>
                            <th>Blood Group</th>
                            <th>Phone</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($searchResults as $patient): ?>
                            <tr>
                                <td><strong><?= e($patient['patient_number']) ?></strong></td>
                                <td><?= e($patient['full_name']) ?></td>
                                <td><?= $patient['date_of_birth'] ? e(date('M j, Y', strtotime($patient['date_of_birth']))) : 'N/A' ?></td>
                                <td><?= e($patient['gender'] ?? 'N/A') ?></td>
                                <td><?= e($patient['blood_group'] ?? 'N/A') ?></td>
                                <td><?= e($patient['phone'] ?? 'N/A') ?></td>
                                <td>
                                    <a href="?action=consultation&sub=consult&patient_id=<?= $patient['id'] ?>" class="btn btn-primary" style="padding: 0.5rem 1rem; font-size: 0.875rem;">
                                        Start Consultation
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    <?php elseif (isset($_GET['search'])): ?>
        <div class="card">
            <p style="text-align: center; padding: 2rem; color: var(--text-light);">
                No patients found matching your search. Please try again.
            </p>
        </div>
    <?php endif; ?>

<?php elseif ($subAction === 'consult' && $selectedPatient): ?>
    <!-- Patient Consultation Form -->
    <div class="card">
        <div style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 1rem; margin-bottom: 1.5rem;">
            <h2>Patient Consultation</h2>
            <a href="?action=consultation" class="btn btn-secondary">← Back to Search</a>
        </div>

        <!-- Patient Info -->
        <div style="background: var(--light-color); padding: 1.5rem; border-radius: 8px; margin-bottom: 2rem;">
            <h3 style="margin-top: 0;">Patient Information</h3>
            <div class="grid">
                <div>
                    <p><strong>Patient Number:</strong> <?= e($selectedPatient['patient_number']) ?></p>
                    <p><strong>Full Name:</strong> <?= e($selectedPatient['full_name']) ?></p>
                    <p><strong>Age:</strong> <?= $selectedPatient['date_of_birth'] ? floor((time() - strtotime($selectedPatient['date_of_birth'])) / 31556926) : 'N/A' ?> years</p>
                </div>
                <div>
                    <p><strong>Gender:</strong> <?= e($selectedPatient['gender'] ?? 'N/A') ?></p>
                    <p><strong>Blood Group:</strong> <?= e($selectedPatient['blood_group'] ?? 'N/A') ?></p>
                    <p><strong>Phone:</strong> <?= e($selectedPatient['phone'] ?? 'N/A') ?></p>
                </div>
            </div>
        </div>

        <!-- Medical History -->
        <?php if (isset($medicalHistory) && count($medicalHistory) > 0): ?>
            <div style="margin-bottom: 2rem;">
                <h3>Recent Consultations</h3>
                <div style="overflow-x: auto;">
                    <table class="table" style="font-size: 0.9rem;">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Doctor</th>
                                <th>Chief Complaint</th>
                                <th>Diagnosis</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($medicalHistory as $record): ?>
                                <tr>
                                    <td><?= e(date('M j, Y g:i A', strtotime($record['created_at']))) ?></td>
                                    <td><?= e($record['doctor_name']) ?></td>
                                    <td><?= e($record['chief_complaint'] ?? 'N/A') ?></td>
                                    <td><?= e($record['diagnosis'] ?? 'N/A') ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        <?php endif; ?>

        <!-- Consultation Form -->
        <form method="POST" action="?action=consultation&sub=consult&patient_id=<?= $selectedPatient['id'] ?>">
            <input type="hidden" name="patient_id" value="<?= $selectedPatient['id'] ?>">

            <h3>Examination Details</h3>
            <div class="form-group">
                <label for="chief_complaint">Chief Complaint *</label>
                <textarea id="chief_complaint" name="chief_complaint" class="form-control" rows="3" required
                          placeholder="Patient's main complaint or reason for visit"></textarea>
            </div>

            <div class="form-group">
                <label for="diagnosis">Diagnosis *</label>
                <textarea id="diagnosis" name="diagnosis" class="form-control" rows="3" required
                          placeholder="Diagnosis based on examination"></textarea>
            </div>

            <div class="form-group">
                <label for="notes">Clinical Notes</label>
                <textarea id="notes" name="notes" class="form-control" rows="3"
                          placeholder="Additional clinical observations, examination findings"></textarea>
            </div>

            <hr style="margin: 2rem 0;">

            <!-- Prescription Section -->
            <h3>Prescribe Medications</h3>
            <p style="color: var(--text-dark); margin-bottom: 1rem;">Add drugs to prescribe. Leave blank if no medication needed.</p>

            <div id="drugs-container">
                <?php for ($i = 0; $i < 3; $i++): ?>
                    <div class="drug-entry" style="background: var(--light-color); padding: 1rem; border-radius: 8px; margin-bottom: 1rem;">
                        <div class="grid">
                            <div class="form-group" style="flex: 2;">
                                <label>Drug *</label>
                                <select name="drugs[]" class="form-control drug-select">
                                    <option value="">Select Drug</option>
                                    <?php foreach ($drugs as $drug): ?>
                                        <option value="<?= $drug['id'] ?>" data-price="<?= $drug['selling_price'] ?>">
                                            <?= e($drug['drug_name']) ?> (Stock: <?= $drug['quantity'] ?>)
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <input type="hidden" name="drug_names[]" class="drug-name">
                            </div>
                            <div class="form-group">
                                <label>Dosage *</label>
                                <input type="text" name="dosages[]" class="form-control" placeholder="e.g., 500mg">
                            </div>
                            <div class="form-group">
                                <label>Frequency *</label>
                                <input type="text" name="frequencies[]" class="form-control" placeholder="e.g., 3x daily">
                            </div>
                            <div class="form-group">
                                <label>Duration *</label>
                                <input type="text" name="durations[]" class="form-control" placeholder="e.g., 7 days">
                            </div>
                            <div class="form-group">
                                <label>Quantity *</label>
                                <input type="number" name="quantities[]" class="form-control" placeholder="e.g., 21" min="1">
                            </div>
                        </div>
                        <div class="form-group" style="margin-top: 1rem;">
                            <label>Instructions</label>
                            <input type="text" name="instructions[]" class="form-control" placeholder="e.g., Take after meals">
                        </div>
                    </div>
                <?php endfor; ?>
            </div>

            <button type="button" onclick="addDrugRow()" class="btn btn-secondary" style="margin-bottom: 1.5rem;">+ Add Another Drug</button>

            <hr style="margin: 2rem 0;">

            <!-- Lab Tests Section -->
            <h3>Request Lab Tests</h3>
            <p style="color: var(--text-dark); margin-bottom: 1rem;">Select lab tests to request. Leave blank if no tests needed.</p>

            <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(250px, 1fr)); gap: 1rem; margin-bottom: 1rem;">
                <?php foreach ($labTests as $test): ?>
                    <label style="display: flex; align-items: center; gap: 0.5rem; padding: 0.75rem; background: var(--light-color); border-radius: 6px; cursor: pointer;">
                        <input type="checkbox" name="lab_tests[]" value="<?= $test['id'] ?>">
                        <div>
                            <strong><?= e($test['test_name']) ?></strong>
                            <div style="font-size: 0.875rem; color: var(--text-dark);">Cost: <?= formatCurrency($test['cost']) ?></div>
                        </div>
                    </label>
                <?php endforeach; ?>
            </div>

            <div class="form-group">
                <label for="lab_notes">Lab Test Notes</label>
                <textarea id="lab_notes" name="lab_notes" class="form-control" rows="2"
                          placeholder="Any special instructions for lab"></textarea>
            </div>

            <hr style="margin: 2rem 0;">

            <button type="submit" name="save_consultation" class="btn btn-primary">Save Consultation & Generate Prescriptions/Requests</button>
            <a href="?action=consultation" class="btn btn-secondary">Cancel</a>
        </form>
    </div>
<?php endif; ?>

<script>
function addDrugRow() {
    const container = document.getElementById('drugs-container');
    const newRow = document.createElement('div');
    newRow.className = 'drug-entry';
    newRow.style.cssText = 'background: var(--light-color); padding: 1rem; border-radius: 8px; margin-bottom: 1rem;';

    // Get drugs from first select
    const drugOptions = document.querySelector('.drug-select').innerHTML;

    newRow.innerHTML = `
        <div class="grid">
            <div class="form-group" style="flex: 2;">
                <label>Drug *</label>
                <select name="drugs[]" class="form-control drug-select" onchange="updateDrugName(this)">
                    ${drugOptions}
                </select>
                <input type="hidden" name="drug_names[]" class="drug-name">
            </div>
            <div class="form-group">
                <label>Dosage *</label>
                <input type="text" name="dosages[]" class="form-control" placeholder="e.g., 500mg">
            </div>
            <div class="form-group">
                <label>Frequency *</label>
                <input type="text" name="frequencies[]" class="form-control" placeholder="e.g., 3x daily">
            </div>
            <div class="form-group">
                <label>Duration *</label>
                <input type="text" name="durations[]" class="form-control" placeholder="e.g., 7 days">
            </div>
            <div class="form-group">
                <label>Quantity *</label>
                <input type="number" name="quantities[]" class="form-control" placeholder="e.g., 21" min="1">
            </div>
        </div>
        <div class="form-group" style="margin-top: 1rem;">
            <label>Instructions</label>
            <input type="text" name="instructions[]" class="form-control" placeholder="e.g., Take after meals">
        </div>
        <button type="button" onclick="this.parentElement.remove()" class="btn btn-secondary" style="margin-top: 1rem; font-size: 0.875rem;">Remove</button>
    `;

    container.appendChild(newRow);
}

function updateDrugName(select) {
    const nameInput = select.parentElement.querySelector('.drug-name');
    const selectedOption = select.options[select.selectedIndex];
    nameInput.value = selectedOption.text;
}

// Initialize drug name inputs
document.querySelectorAll('.drug-select').forEach(select => {
    select.addEventListener('change', function() {
        updateDrugName(this);
    });
});
</script>
