<?php
$db = Database::getInstance()->getConnection();
$tenantId = $_SESSION['tenant_id'];
$userId = $_SESSION['user_id'];
$userRole = $_SESSION['role'];

// Check if user can access lab requests
$canAccessLab = in_array($userRole, ['hospital_admin', 'lab_scientist', 'lab_technician']);

if (!$canAccessLab) {
    header('Location: ?action=dashboard');
    exit;
}

$subAction = $_GET['sub'] ?? 'list';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['process_tests'])) {
        // Generate invoice for requested lab tests
        $labRequestId = $_POST['lab_request_id'];
        $totalAmount = 0;

        // Get lab request details
        $stmt = $db->prepare("
            SELECT lr.*, p.full_name as patient_name, p.patient_number
            FROM lab_requests lr
            JOIN patients p ON lr.patient_id = p.id
            WHERE lr.id = ? AND lr.tenant_id = ?
        ");
        $stmt->execute([$labRequestId, $tenantId]);
        $labRequest = $stmt->fetch(PDO::FETCH_ASSOC);

        // Generate invoice number
        $invoiceNumber = 'INV' . date('Ymd') . str_pad($db->query("SELECT COUNT(*) FROM billing WHERE tenant_id = $tenantId")->fetchColumn() + 1, 4, '0', STR_PAD_LEFT);

        // Get selected tests to process
        $processedTests = [];
        $testDescriptions = [];

        if (isset($_POST['tests_to_process'])) {
            foreach ($_POST['tests_to_process'] as $itemId) {
                // Get test details with cost
                $itemStmt = $db->prepare("
                    SELECT lri.*, lt.cost
                    FROM lab_request_items lri
                    LEFT JOIN lab_tests lt ON lri.test_id = lt.id
                    WHERE lri.id = ? AND lri.tenant_id = ?
                ");
                $itemStmt->execute([$itemId, $tenantId]);
                    $item = $itemStmt->fetch(PDO::FETCH_ASSOC);

                if ($item && $item['cost']) {
                    $totalAmount += $item['cost'];
                    $processedTests[] = $item;
                    $testDescriptions[] = "{$item['test_name']} - " . formatCurrency($item['cost']);
                }

                // Update item status to 'invoiced'
                $updateStmt = $db->prepare("
                    UPDATE lab_request_items
                    SET status = 'invoiced'
                    WHERE id = ?
                ");
                $updateStmt->execute([$itemId]);
            }
        }

        // Create billing invoice
        if ($totalAmount > 0) {
            $description = "Laboratory - Request #{$labRequest['request_number']}: " . implode(', ', $testDescriptions);

            $stmt = $db->prepare("
                INSERT INTO billing (tenant_id, patient_id, invoice_number, description, amount, payment_status)
                VALUES (?, ?, ?, ?, ?, 'pending')
            ");
            $stmt->execute([
                $tenantId,
                $labRequest['patient_id'],
                $invoiceNumber,
                $description,
                $totalAmount
            ]);

            $invoiceId = $db->lastInsertId();

            // Update items with invoice ID
            foreach ($_POST['tests_to_process'] as $itemId) {
                $updateStmt = $db->prepare("
                    UPDATE lab_request_items
                    SET invoice_id = ?
                    WHERE id = ?
                ");
                $updateStmt->execute([$invoiceId, $itemId]);
            }

            // Update lab request status
            $stmt = $db->prepare("
                UPDATE lab_requests
                SET status = 'invoiced'
                WHERE id = ?
            ");
            $stmt->execute([$labRequestId]);

            $message = "Lab tests invoiced! Invoice {$invoiceNumber} created for " . formatCurrency($totalAmount);
        } else {
            $error = "No tests were selected for processing or no pricing available.";
        }

        $subAction = 'list';
    } elseif (isset($_POST['save_results'])) {
        // Save lab test results
        $labRequestId = $_POST['lab_request_id'];

        foreach ($_POST['results'] as $itemId => $result) {
            if (!empty($result) || !empty($_POST['reference_ranges'][$itemId]) || !empty($_POST['test_notes'][$itemId])) {
                $stmt = $db->prepare("
                    UPDATE lab_request_items
                    SET result = ?, reference_range = ?, notes = ?, status = 'completed', performed_by = ?, result_date = CURRENT_TIMESTAMP
                    WHERE id = ? AND tenant_id = ?
                ");
                $stmt->execute([
                    $result,
                    $_POST['reference_ranges'][$itemId],
                    $_POST['test_notes'][$itemId],
                    $userId,
                    $itemId,
                    $tenantId
                ]);
            }
        }

        // Update lab request status
        $stmt = $db->prepare("
            UPDATE lab_requests
            SET status = 'completed'
            WHERE id = ?
        ");
        $stmt->execute([$labRequestId]);

        $message = "Lab test results saved successfully!";
        $subAction = 'list';
    }
}

// Get all pending lab requests
$pendingRequests = $db->query("
    SELECT lr.*, p.full_name as patient_name, p.patient_number,
           u.full_name as doctor_name,
           COUNT(lri.id) as test_count
    FROM lab_requests lr
    JOIN patients p ON lr.patient_id = p.id
    LEFT JOIN users u ON lr.doctor_id = u.id
    LEFT JOIN lab_request_items lri ON lr.id = lri.lab_request_id
    WHERE lr.tenant_id = $tenantId AND lr.status IN ('pending', 'invoiced')
    GROUP BY lr.id
    ORDER BY lr.created_at DESC
")->fetchAll(PDO::FETCH_ASSOC);

// Get lab request details if viewing
$viewRequest = null;
$requestTests = [];

if ($subAction === 'view' && isset($_GET['id'])) {
    $stmt = $db->prepare("
        SELECT lr.*, p.full_name as patient_name, p.patient_number, p.phone,
               u.full_name as doctor_name
        FROM lab_requests lr
        JOIN patients p ON lr.patient_id = p.id
        LEFT JOIN users u ON lr.doctor_id = u.id
        WHERE lr.id = ? AND lr.tenant_id = ?
    ");
    $stmt->execute([$_GET['id'], $tenantId]);
    $viewRequest = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($viewRequest) {
        $stmt = $db->prepare("
            SELECT lri.*, lt.cost, lt.reference_range as test_reference_range
            FROM lab_request_items lri
            LEFT JOIN lab_tests lt ON lri.test_id = lt.id
            WHERE lri.lab_request_id = ? AND lri.tenant_id = ?
            ORDER BY lri.id
        ");
        $stmt->execute([$_GET['id'], $tenantId]);
        $requestTests = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
?>

<div class="hero">
    <h1>Lab Test Requests</h1>
    <p>View, invoice, and process lab test requests</p>
</div>

<?php if (isset($message)): ?>
    <div class="alert alert-success"><?= e($message) ?></div>
<?php endif; ?>

<?php if (isset($error)): ?>
    <div class="alert alert-error"><?= e($error) ?></div>
<?php endif; ?>

<?php if ($subAction === 'view' && $viewRequest): ?>
    <!-- Lab Request Details -->
    <div class="card">
        <div style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 1rem; margin-bottom: 1.5rem;">
            <h2>Lab Request #<?= e($viewRequest['request_number']) ?></h2>
            <a href="?action=lab_requests" class="btn btn-secondary">← Back to List</a>
        </div>

        <!-- Patient & Doctor Info -->
        <div style="background: var(--light-color); padding: 1.5rem; border-radius: 8px; margin-bottom: 2rem;">
            <div class="grid">
                <div>
                    <p><strong>Patient:</strong> <?= e($viewRequest['patient_name']) ?> (<?= e($viewRequest['patient_number']) ?>)</p>
                    <p><strong>Phone:</strong> <?= e($viewRequest['phone'] ?? 'N/A') ?></p>
                </div>
                <div>
                    <p><strong>Requesting Doctor:</strong> <?= e($viewRequest['doctor_name']) ?></p>
                    <p><strong>Date:</strong> <?= e(date('M j, Y g:i A', strtotime($viewRequest['created_at']))) ?></p>
                </div>
            </div>
            <?php if ($viewRequest['notes']): ?>
                <div style="margin-top: 1rem; padding-top: 1rem; border-top: 1px solid #ddd;">
                    <p><strong>Doctor's Notes:</strong> <?= e($viewRequest['notes']) ?></p>
                </div>
            <?php endif; ?>
        </div>

        <?php if ($viewRequest['status'] === 'pending'): ?>
            <!-- Generate Invoice Form -->
            <form method="POST" action="?action=lab_requests&sub=view&id=<?= $viewRequest['id'] ?>">
                <input type="hidden" name="lab_request_id" value="<?= $viewRequest['id'] ?>">

                <h3>Tests to Invoice</h3>

                <?php if (count($requestTests) > 0): ?>
                    <div style="overflow-x: auto; margin-bottom: 1.5rem;">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th style="width: 40px;">Process</th>
                                    <th>Test Name</th>
                                    <th>Reference Range</th>
                                    <th>Cost</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php
                                $requestTotal = 0;
                                foreach ($requestTests as $test):
                                    $requestTotal += $test['cost'] ?? 0;
                                ?>
                                    <tr>
                                        <td>
                                            <?php if ($test['status'] === 'pending'): ?>
                                                <input type="checkbox" name="tests_to_process[]" value="<?= $test['id'] ?>" checked>
                                            <?php else: ?>
                                                <span style="color: green;">✓</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><strong><?= e($test['test_name']) ?></strong></td>
                                        <td><?= e($test['test_reference_range'] ?? 'N/A') ?></td>
                                        <td><?= formatCurrency($test['cost'] ?? 0) ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                            <tfoot>
                                <tr style="background-color: var(--primary-color); color: white;">
                                    <td colspan="3" style="text-align: right; padding: 1rem;"><strong>Total:</strong></td>
                                    <td style="padding: 1rem;"><strong><?= formatCurrency($requestTotal) ?></strong></td>
                                </tr>
                            </tfoot>
                        </table>
                    </div>

                    <div class="alert alert-info" style="margin-bottom: 1.5rem;">
                        <strong>Note:</strong> Select tests that can be performed and click "Generate Invoice". The patient will take the invoice to accounting for payment. After payment, enter test results.
                    </div>

                    <?php $hasPendingTests = false; ?>
                    <?php foreach ($requestTests as $test): ?>
                        <?php if ($test['status'] === 'pending'): ?>
                            <?php $hasPendingTests = true; break; ?>
                        <?php endif; ?>
                    <?php endforeach; ?>

                    <?php if ($hasPendingTests): ?>
                        <button type="submit" name="process_tests" class="btn btn-primary">Generate Invoice</button>
                    <?php endif; ?>
                <?php else: ?>
                    <p style="text-align: center; padding: 2rem; color: var(--text-light);">
                        No tests found in this request.
                    </p>
                <?php endif; ?>

                <a href="?action=lab_requests" class="btn btn-secondary">Back to List</a>
            </form>

        <?php elseif ($viewRequest['status'] === 'invoiced'): ?>
            <!-- Enter Results Form -->
            <form method="POST" action="?action=lab_requests&sub=view&id=<?= $viewRequest['id'] ?>">
                <input type="hidden" name="lab_request_id" value="<?= $viewRequest['id'] ?>">

                <h3>Enter Test Results</h3>

                <?php if (count($requestTests) > 0): ?>
                    <div style="overflow-x: auto; margin-bottom: 1.5rem;">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Test Name</th>
                                    <th>Reference Range</th>
                                    <th>Result *</th>
                                    <th>Notes</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($requestTests as $test): ?>
                                    <tr>
                                        <td><strong><?= e($test['test_name']) ?></strong></td>
                                        <td><?= e($test['test_reference_range'] ?? 'N/A') ?></td>
                                        <td>
                                            <?php if ($test['status'] === 'completed'): ?>
                                                <span style="font-weight: bold;"><?= e($test['result']) ?></span>
                                            <?php else: ?>
                                                <input type="text" name="results[<?= $test['id'] ?>]" class="form-control"
                                                       placeholder="Enter result" required>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($test['status'] === 'completed'): ?>
                                                <?= e($test['notes'] ?? '-') ?>
                                            <?php else: ?>
                                                <input type="text" name="test_notes[<?= $test['id'] ?>]" class="form-control"
                                                       placeholder="Additional notes">
                                                <input type="hidden" name="reference_ranges[<?= $test['id'] ?>]" value="<?= e($test['test_reference_range'] ?? '') ?>">
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                    <div class="alert alert-info" style="margin-bottom: 1.5rem;">
                        <strong>Note:</strong> Enter test results accurately. Results will be visible to the requesting doctor and in patient's medical records.
                    </div>

                    <?php $hasPendingTests = false; ?>
                    <?php foreach ($requestTests as $test): ?>
                        <?php if ($test['status'] !== 'completed'): ?>
                            <?php $hasPendingTests = true; break; ?>
                        <?php endif; ?>
                    <?php endforeach; ?>

                    <?php if ($hasPendingTests): ?>
                        <button type="submit" name="save_results" class="btn btn-primary">Save Results</button>
                    <?php endif; ?>
                <?php else: ?>
                    <p style="text-align: center; padding: 2rem; color: var(--text-light);">
                        No tests found in this request.
                    </p>
                <?php endif; ?>

                <a href="?action=lab_requests" class="btn btn-secondary">Back to List</a>
            </form>

        <?php else: ?>
            <!-- View Completed Results -->
            <h3>Test Results</h3>

            <?php if (count($requestTests) > 0): ?>
                <div style="overflow-x: auto; margin-bottom: 1.5rem;">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Test Name</th>
                                <th>Reference Range</th>
                                <th>Result</th>
                                <th>Notes</th>
                                <th>Performed By</th>
                                <th>Date</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($requestTests as $test): ?>
                                <tr>
                                    <td><strong><?= e($test['test_name']) ?></strong></td>
                                    <td><?= e($test['test_reference_range'] ?? 'N/A') ?></td>
                                    <td style="font-weight: bold; font-size: 1.1rem;"><?= e($test['result']) ?></td>
                                    <td><?= e($test['notes'] ?? '-') ?></td>
                                    <td>
                                        <?php
                                        $userStmt = $db->prepare("SELECT full_name FROM users WHERE id = ?");
                                        $userStmt->execute([$test['performed_by']]);
                                        echo e($userStmt->fetchColumn() ?? 'N/A');
                                        ?>
                                    </td>
                                    <td><?= $test['result_date'] ? e(date('M j, Y g:i A', strtotime($test['result_date']))) : 'N/A' ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>

            <a href="?action=lab_requests" class="btn btn-secondary">Back to List</a>
        <?php endif; ?>
    </div>

<?php else: ?>
    <!-- Pending Lab Requests List -->
    <div class="card">
        <h2>Pending Lab Requests (<?= count($pendingRequests) ?>)</h2>

        <?php if (count($pendingRequests) > 0): ?>
            <div style="overflow-x: auto;">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Request #</th>
                            <th>Patient</th>
                            <th>Patient Number</th>
                            <th>Doctor</th>
                            <th>Tests</th>
                            <th>Status</th>
                            <th>Date</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($pendingRequests as $request): ?>
                            <tr>
                                <td><strong><?= e($request['request_number']) ?></strong></td>
                                <td><?= e($request['patient_name']) ?></td>
                                <td><?= e($request['patient_number']) ?></td>
                                <td><?= e($request['doctor_name']) ?></td>
                                <td><?= $request['test_count'] ?></td>
                                <td>
                                    <span style="padding: 0.25rem 0.75rem; border-radius: 4px; font-size: 0.875rem;
                                        background-color: <?= $request['status'] === 'pending' ? '#fef3c7' : '#dbeafe' ?>;
                                        color: <?= $request['status'] === 'pending' ? '#92400e' : '#1e40af' ?>;">
                                        <?= ucfirst($request['status']) ?>
                                    </span>
                                </td>
                                <td><?= e(date('M j, Y g:i A', strtotime($request['created_at']))) ?></td>
                                <td>
                                    <a href="?action=lab_requests&sub=view&id=<?= $request['id'] ?>" class="btn btn-primary" style="padding: 0.5rem 1rem; font-size: 0.875rem;">
                                        <?= $request['status'] === 'pending' ? 'Process' : 'View' ?>
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <p style="text-align: center; padding: 3rem; color: var(--text-light);">
                No pending lab requests to process.
            </p>
        <?php endif; ?>
    </div>
<?php endif; ?>
