<?php
$db = Database::getInstance()->getConnection();
$tenantId = $_SESSION['tenant_id'];
$userRole = $_SESSION['role'];
$subAction = $_GET['sub'] ?? 'list';

// Check if user can manage lab data
$canManageLab = in_array($userRole, ['hospital_admin', 'lab_scientist', 'lab_technician']);

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $canManageLab) {
    if (isset($_POST['add_test'])) {
        $testCode = 'TEST' . str_pad($db->query("SELECT COUNT(*) FROM lab_tests WHERE tenant_id = $tenantId")->fetchColumn() + 1, 4, '0', STR_PAD_LEFT);

        $stmt = $db->prepare("
            INSERT INTO lab_tests (tenant_id, test_name, test_code, category, cost, normal_range, description)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");

        $stmt->execute([
            $tenantId,
            $_POST['test_name'],
            $testCode,
            $_POST['category'],
            $_POST['cost'],
            $_POST['normal_range'],
            $_POST['description']
        ]);

        $message = 'Lab test added successfully! Test Code: ' . $testCode;
        $subAction = 'list';
    } elseif (isset($_POST['update_test'])) {
        $stmt = $db->prepare("
            UPDATE lab_tests SET
                test_name = ?, category = ?, cost = ?, normal_range = ?, description = ?, status = ?
            WHERE id = ? AND tenant_id = ?
        ");

        $stmt->execute([
            $_POST['test_name'],
            $_POST['category'],
            $_POST['cost'],
            $_POST['normal_range'],
            $_POST['description'],
            $_POST['status'],
            $_POST['test_id'],
            $tenantId
        ]);

        $message = 'Lab test updated successfully!';
        $subAction = 'list';
    } elseif (isset($_POST['add_order'])) {
        $orderNumber = 'LO' . date('Ymd') . str_pad($db->query("SELECT COUNT(*) FROM lab_orders WHERE tenant_id = $tenantId")->fetchColumn() + 1, 4, '0', STR_PAD_LEFT);

        // Get test cost
        $stmt = $db->prepare("SELECT cost FROM lab_tests WHERE id = ? AND tenant_id = ?");
        $stmt->execute([$_POST['test_id'], $tenantId]);
        $testCost = $stmt->fetchColumn();

        $stmt = $db->prepare("
            INSERT INTO lab_orders (tenant_id, patient_id, doctor_id, test_id, order_number, cost, payment_status, status)
            VALUES (?, ?, ?, ?, ?, ?, 'pending', 'pending')
        ");

        $stmt->execute([
            $tenantId,
            $_POST['patient_id'],
            $_POST['doctor_id'],
            $_POST['test_id'],
            $orderNumber,
            $testCost
        ]);

        $message = 'Lab order created successfully! Order #: ' . $orderNumber;
        $subAction = 'orders';
    } elseif (isset($_POST['update_order'])) {
        $resultDate = $_POST['status'] === 'completed' ? date('Y-m-d H:i:s') : null;

        $stmt = $db->prepare("
            UPDATE lab_orders SET
                sample_collected_date = ?, result = ?, interpretation = ?, status = ?,
                result_date = ?, tested_by = ?
            WHERE id = ? AND tenant_id = ?
        ");

        $stmt->execute([
            $_POST['sample_collected_date'],
            $_POST['result'],
            $_POST['interpretation'],
            $_POST['status'],
            $resultDate,
            $_SESSION['user_id'],
            $_POST['order_id'],
            $tenantId
        ]);

        $message = 'Lab order updated successfully!';
        $subAction = 'orders';
    } elseif (isset($_POST['generate_invoice'])) {
        // Generate invoice number
        $invoiceNumber = 'INV' . date('Ymd') . str_pad($db->query("SELECT COUNT(*) FROM billing WHERE tenant_id = $tenantId")->fetchColumn() + 1, 4, '0', STR_PAD_LEFT);

        // Get order details
        $stmt = $db->prepare("
            SELECT lo.*, lt.test_name, p.full_name as patient_name
            FROM lab_orders lo
            JOIN lab_tests lt ON lo.test_id = lt.id
            JOIN patients p ON lo.patient_id = p.id
            WHERE lo.id = ? AND lo.tenant_id = ?
        ");
        $stmt->execute([$_POST['order_id'], $tenantId]);
        $order = $stmt->fetch(PDO::FETCH_ASSOC);

        // Create billing entry
        $stmt = $db->prepare("
            INSERT INTO billing (tenant_id, patient_id, invoice_number, description, amount, payment_status)
            VALUES (?, ?, ?, ?, ?, 'pending')
        ");

        $stmt->execute([
            $tenantId,
            $order['patient_id'],
            $invoiceNumber,
            'Laboratory Test: ' . $order['test_name'],
            $order['cost']
        ]);

        // Update lab order payment status
        $stmt = $db->prepare("UPDATE lab_orders SET payment_status = 'invoiced' WHERE id = ? AND tenant_id = ?");
        $stmt->execute([$_POST['order_id'], $tenantId]);

        $message = 'Invoice generated successfully! Invoice #: ' . $invoiceNumber;
        $subAction = 'orders';
    } elseif (isset($_POST['delete_test'])) {
        $stmt = $db->prepare("DELETE FROM lab_tests WHERE id = ? AND tenant_id = ?");
        $stmt->execute([$_POST['test_id'], $tenantId]);
        $message = 'Lab test deleted successfully!';
        $subAction = 'list';
    }
}

// Get test for editing
$editTest = null;
if ($subAction === 'edit_test' && isset($_GET['id'])) {
    $stmt = $db->prepare("SELECT * FROM lab_tests WHERE id = ? AND tenant_id = ?");
    $stmt->execute([$_GET['id'], $tenantId]);
    $editTest = $stmt->fetch(PDO::FETCH_ASSOC);
}

// Get order for editing
$editOrder = null;
if ($subAction === 'edit_order' && isset($_GET['id'])) {
    $stmt = $db->prepare("
        SELECT lo.*, p.full_name as patient_name, d.full_name as doctor_name, lt.test_name
        FROM lab_orders lo
        JOIN patients p ON lo.patient_id = p.id
        JOIN doctors d ON lo.doctor_id = d.id
        JOIN lab_tests lt ON lo.test_id = lt.id
        WHERE lo.id = ? AND lo.tenant_id = ?
    ");
    $stmt->execute([$_GET['id'], $tenantId]);
    $editOrder = $stmt->fetch(PDO::FETCH_ASSOC);
}

// Get lab tests
$labTests = $db->query("SELECT * FROM lab_tests WHERE tenant_id = $tenantId ORDER BY test_name")->fetchAll(PDO::FETCH_ASSOC);

// Get lab orders
$labOrders = $db->query("
    SELECT lo.*, p.full_name as patient_name, p.patient_number, d.full_name as doctor_name, lt.test_name, lt.test_code
    FROM lab_orders lo
    JOIN patients p ON lo.patient_id = p.id
    JOIN doctors d ON lo.doctor_id = d.id
    JOIN lab_tests lt ON lo.test_id = lt.id
    WHERE lo.tenant_id = $tenantId
    ORDER BY lo.order_date DESC
")->fetchAll(PDO::FETCH_ASSOC);

// Get patients and doctors for dropdowns
$patients = $db->query("SELECT id, patient_number, full_name FROM patients WHERE tenant_id = $tenantId AND status = 'active' ORDER BY full_name")->fetchAll(PDO::FETCH_ASSOC);
$doctors = $db->query("SELECT id, doctor_number, full_name FROM doctors WHERE tenant_id = $tenantId AND status = 'active' ORDER BY full_name")->fetchAll(PDO::FETCH_ASSOC);

// Statistics
$stats = [
    'total_tests' => count($labTests),
    'total_orders' => count($labOrders),
    'pending' => $db->query("SELECT COUNT(*) FROM lab_orders WHERE tenant_id = $tenantId AND status = 'pending'")->fetchColumn(),
    'completed' => $db->query("SELECT COUNT(*) FROM lab_orders WHERE tenant_id = $tenantId AND status = 'completed'")->fetchColumn(),
    'revenue' => $db->query("SELECT SUM(cost) FROM lab_orders WHERE tenant_id = $tenantId AND payment_status = 'paid'")->fetchColumn() ?? 0,
];
?>

<div class="hero">
    <h1>Laboratory Management</h1>
    <p>Manage lab tests, orders, and results</p>
</div>

<?php if (isset($message)): ?>
    <div class="alert alert-success"><?= e($message) ?></div>
<?php endif; ?>

<div class="stats">
    <div class="stat-card">
        <div class="stat-value"><?= $stats['total_tests'] ?></div>
        <div class="stat-label">Available Tests</div>
    </div>
    <div class="stat-card">
        <div class="stat-value"><?= $stats['total_orders'] ?></div>
        <div class="stat-label">Total Orders</div>
    </div>
    <div class="stat-card warning">
        <div class="stat-value"><?= $stats['pending'] ?></div>
        <div class="stat-label">Pending Tests</div>
    </div>
    <div class="stat-card success">
        <div class="stat-value"><?= $stats['completed'] ?></div>
        <div class="stat-label">Completed</div>
    </div>
    <div class="stat-card success">
        <div class="stat-value"><?= formatCurrency($stats['revenue']) ?></div>
        <div class="stat-label">Revenue (Paid)</div>
    </div>
</div>

<!-- Navigation Tabs -->
<div class="card">
    <div style="display: flex; gap: 0.5rem; margin-bottom: 1.5rem; flex-wrap: wrap;">
        <a href="?action=laboratory&sub=list" class="btn <?= $subAction === 'list' ? 'btn-primary' : 'btn-secondary' ?>">Lab Tests Catalog</a>
        <a href="?action=laboratory&sub=orders" class="btn <?= $subAction === 'orders' ? 'btn-primary' : 'btn-secondary' ?>">Test Orders</a>
    </div>
</div>

<?php if ($subAction === 'add_test' && $canManageLab): ?>
    <!-- Add Lab Test Form -->
    <div class="card">
        <h2>Add New Lab Test</h2>
        <form method="POST" action="?action=laboratory">
            <div class="grid">
                <div class="form-group">
                    <label for="test_name">Test Name *</label>
                    <input type="text" id="test_name" name="test_name" class="form-control" required>
                </div>

                <div class="form-group">
                    <label for="category">Category *</label>
                    <input type="text" id="category" name="category" class="form-control" required
                           placeholder="e.g., Hematology, Chemistry, Microbiology">
                </div>

                <div class="form-group">
                    <label for="cost">Cost ($) *</label>
                    <input type="number" id="cost" name="cost" class="form-control" required step="0.01" min="0">
                </div>

                <div class="form-group">
                    <label for="normal_range">Normal Range</label>
                    <input type="text" id="normal_range" name="normal_range" class="form-control"
                           placeholder="e.g., 4.5-5.5 million cells/mcL">
                </div>
            </div>

            <div class="form-group">
                <label for="description">Description</label>
                <textarea id="description" name="description" class="form-control" rows="3"></textarea>
            </div>

            <div style="display: flex; gap: 1rem;">
                <button type="submit" name="add_test" class="btn btn-primary">Add Test</button>
                <a href="?action=laboratory&sub=list" class="btn btn-secondary">Cancel</a>
            </div>
        </form>
    </div>

<?php elseif ($subAction === 'edit_test' && $editTest && $canManageLab): ?>
    <!-- Edit Lab Test Form -->
    <div class="card">
        <h2>Edit Lab Test</h2>
        <form method="POST" action="?action=laboratory">
            <input type="hidden" name="test_id" value="<?= e($editTest['id']) ?>">

            <div class="grid">
                <div class="form-group">
                    <label for="test_name">Test Name *</label>
                    <input type="text" id="test_name" name="test_name" class="form-control"
                           value="<?= e($editTest['test_name']) ?>" required>
                </div>

                <div class="form-group">
                    <label for="category">Category *</label>
                    <input type="text" id="category" name="category" class="form-control"
                           value="<?= e($editTest['category']) ?>" required>
                </div>

                <div class="form-group">
                    <label for="cost">Cost ($) *</label>
                    <input type="number" id="cost" name="cost" class="form-control"
                           value="<?= e($editTest['cost']) ?>" required step="0.01" min="0">
                </div>

                <div class="form-group">
                    <label for="normal_range">Normal Range</label>
                    <input type="text" id="normal_range" name="normal_range" class="form-control"
                           value="<?= e($editTest['normal_range']) ?>">
                </div>

                <div class="form-group">
                    <label for="status">Status</label>
                    <select id="status" name="status" class="form-control">
                        <option value="active" <?= $editTest['status'] === 'active' ? 'selected' : '' ?>>Active</option>
                        <option value="inactive" <?= $editTest['status'] === 'inactive' ? 'selected' : '' ?>>Inactive</option>
                    </select>
                </div>
            </div>

            <div class="form-group">
                <label for="description">Description</label>
                <textarea id="description" name="description" class="form-control" rows="3"><?= e($editTest['description']) ?></textarea>
            </div>

            <div style="display: flex; gap: 1rem;">
                <button type="submit" name="update_test" class="btn btn-primary">Update Test</button>
                <a href="?action=laboratory&sub=list" class="btn btn-secondary">Cancel</a>
            </div>
        </form>
    </div>

<?php elseif ($subAction === 'add_order' && $canManageLab): ?>
    <!-- Add Lab Order Form -->
    <div class="card">
        <h2>Create New Lab Order</h2>
        <form method="POST" action="?action=laboratory">
            <div class="grid">
                <div class="form-group">
                    <label for="patient_id">Patient *</label>
                    <select id="patient_id" name="patient_id" class="form-control" required>
                        <option value="">Select Patient</option>
                        <?php foreach ($patients as $patient): ?>
                            <option value="<?= $patient['id'] ?>"><?= e($patient['patient_number']) ?> - <?= e($patient['full_name']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label for="doctor_id">Doctor *</label>
                    <select id="doctor_id" name="doctor_id" class="form-control" required>
                        <option value="">Select Doctor</option>
                        <?php foreach ($doctors as $doctor): ?>
                            <option value="<?= $doctor['id'] ?>"><?= e($doctor['full_name']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label for="test_id">Lab Test *</label>
                    <select id="test_id" name="test_id" class="form-control" required>
                        <option value="">Select Test</option>
                        <?php foreach ($labTests as $test): ?>
                            <option value="<?= $test['id'] ?>"><?= e($test['test_name']) ?> - <?= formatCurrency($test['cost']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>

            <div style="display: flex; gap: 1rem; margin-top: 1.5rem;">
                <button type="submit" name="add_order" class="btn btn-primary">Create Order</button>
                <a href="?action=laboratory&sub=orders" class="btn btn-secondary">Cancel</a>
            </div>
        </form>
    </div>

<?php elseif ($subAction === 'edit_order' && $editOrder && $canManageLab): ?>
    <!-- Edit Lab Order Form -->
    <div class="card">
        <h2>Update Lab Order</h2>
        <form method="POST" action="?action=laboratory">
            <input type="hidden" name="order_id" value="<?= e($editOrder['id']) ?>">

            <div class="form-group">
                <label>Order Details</label>
                <div style="padding: 1rem; background: var(--light-color); border-radius: 6px; line-height: 1.8;">
                    <p><strong>Order #:</strong> <?= e($editOrder['order_number']) ?></p>
                    <p><strong>Patient:</strong> <?= e($editOrder['patient_name']) ?> (<?= e($editOrder['patient_number']) ?>)</p>
                    <p><strong>Doctor:</strong> <?= e($editOrder['doctor_name']) ?></p>
                    <p><strong>Test:</strong> <?= e($editOrder['test_name']) ?></p>
                    <p><strong>Cost:</strong> <?= formatCurrency($editOrder['cost']) ?></p>
                </div>
            </div>

            <div class="grid">
                <div class="form-group">
                    <label for="sample_collected_date">Sample Collected Date</label>
                    <input type="datetime-local" id="sample_collected_date" name="sample_collected_date" class="form-control"
                           value="<?= $editOrder['sample_collected_date'] ? date('Y-m-d\TH:i', strtotime($editOrder['sample_collected_date'])) : '' ?>">
                </div>

                <div class="form-group">
                    <label for="status">Status *</label>
                    <select id="status" name="status" class="form-control" required>
                        <option value="pending" <?= $editOrder['status'] === 'pending' ? 'selected' : '' ?>>Pending</option>
                        <option value="processing" <?= $editOrder['status'] === 'processing' ? 'selected' : '' ?>>Processing</option>
                        <option value="completed" <?= $editOrder['status'] === 'completed' ? 'selected' : '' ?>>Completed</option>
                    </select>
                </div>
            </div>

            <div class="form-group">
                <label for="result">Test Result</label>
                <textarea id="result" name="result" class="form-control" rows="4"><?= e($editOrder['result']) ?></textarea>
            </div>

            <div class="form-group">
                <label for="interpretation">Interpretation</label>
                <textarea id="interpretation" name="interpretation" class="form-control" rows="3"><?= e($editOrder['interpretation']) ?></textarea>
            </div>

            <div style="display: flex; gap: 1rem; margin-top: 1.5rem;">
                <button type="submit" name="update_order" class="btn btn-primary">Update Order</button>
                <a href="?action=laboratory&sub=orders" class="btn btn-secondary">Cancel</a>
            </div>
        </form>
    </div>

<?php elseif ($subAction === 'orders'): ?>
    <!-- Lab Orders List -->
    <div class="card">
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1.5rem; flex-wrap: wrap; gap: 1rem;">
            <h2>Lab Test Orders (<?= count($labOrders) ?>)</h2>
            <?php if ($canManageLab): ?>
                <a href="?action=laboratory&sub=add_order" class="btn btn-primary">+ Create Order</a>
            <?php endif; ?>
        </div>

        <?php if (count($labOrders) > 0): ?>
            <div style="overflow-x: auto;">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Order #</th>
                            <th>Patient</th>
                            <th>Doctor</th>
                            <th>Test Name</th>
                            <th>Order Date</th>
                            <th>Result Date</th>
                            <th>Cost</th>
                            <th>Payment</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($labOrders as $order): ?>
                            <tr>
                                <td><strong><?= e($order['order_number']) ?></strong></td>
                                <td><?= e($order['patient_name']) ?><br><small><?= e($order['patient_number']) ?></small></td>
                                <td><?= e($order['doctor_name']) ?></td>
                                <td><?= e($order['test_name']) ?><br><small><?= e($order['test_code']) ?></small></td>
                                <td><?= e(date('M j, Y', strtotime($order['order_date']))) ?></td>
                                <td><?= $order['result_date'] ? e(date('M j, Y', strtotime($order['result_date']))) : '-' ?></td>
                                <td><strong><?= formatCurrency($order['cost']) ?></strong></td>
                                <td>
                                    <span style="padding: 0.25rem 0.75rem; border-radius: 4px; font-size: 0.875rem;
                                        background-color: <?= $order['payment_status'] === 'paid' ? '#d1fae5' : ($order['payment_status'] === 'invoiced' ? '#dbeafe' : '#fef3c7') ?>;
                                        color: <?= $order['payment_status'] === 'paid' ? '#065f46' : ($order['payment_status'] === 'invoiced' ? '#1e40af' : '#92400e') ?>;">
                                        <?= e(ucfirst($order['payment_status'])) ?>
                                    </span>
                                </td>
                                <td>
                                    <span style="padding: 0.25rem 0.75rem; border-radius: 4px; font-size: 0.875rem;
                                        background-color: <?= $order['status'] === 'completed' ? '#d1fae5' : ($order['status'] === 'pending' ? '#fef3c7' : '#dbeafe') ?>;
                                        color: <?= $order['status'] === 'completed' ? '#065f46' : ($order['status'] === 'pending' ? '#92400e' : '#1e40af') ?>;">
                                        <?= e(ucfirst($order['status'])) ?>
                                    </span>
                                </td>
                                <td>
                                    <div style="display: flex; gap: 0.5rem; flex-wrap: wrap;">
                                        <?php if ($canManageLab): ?>
                                            <a href="?action=laboratory&sub=edit_order&id=<?= $order['id'] ?>" class="btn btn-secondary" style="padding: 0.5rem 1rem; font-size: 0.875rem;">Edit</a>
                                            <?php if ($order['payment_status'] === 'pending'): ?>
                                                <form method="POST" action="?action=laboratory&sub=orders" style="display: inline;">
                                                    <input type="hidden" name="order_id" value="<?= $order['id'] ?>">
                                                    <button type="submit" name="generate_invoice" class="btn btn-success" style="padding: 0.5rem 1rem; font-size: 0.875rem;">
                                                        Generate Invoice
                                                    </button>
                                                </form>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <p style="text-align: center; padding: 3rem; color: var(--text-light);">
                No lab orders yet. Click "Create Order" to get started.
            </p>
        <?php endif; ?>
    </div>

<?php else: ?>
    <!-- Lab Tests Catalog -->
    <div class="card">
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1.5rem; flex-wrap: wrap; gap: 1rem;">
            <h2>Lab Tests Catalog (<?= count($labTests) ?>)</h2>
            <?php if ($canManageLab): ?>
                <a href="?action=laboratory&sub=add_test" class="btn btn-primary">+ Add New Test</a>
            <?php endif; ?>
        </div>

        <?php if (count($labTests) > 0): ?>
            <div style="overflow-x: auto;">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Test Code</th>
                            <th>Test Name</th>
                            <th>Category</th>
                            <th>Cost</th>
                            <th>Normal Range</th>
                            <th>Status</th>
                            <?php if ($canManageLab): ?>
                                <th>Actions</th>
                            <?php endif; ?>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($labTests as $test): ?>
                            <tr>
                                <td><strong><?= e($test['test_code']) ?></strong></td>
                                <td><?= e($test['test_name']) ?></td>
                                <td><?= e($test['category']) ?></td>
                                <td><strong><?= formatCurrency($test['cost']) ?></strong></td>
                                <td><?= e($test['normal_range'] ?? 'N/A') ?></td>
                                <td>
                                    <span style="padding: 0.25rem 0.75rem; border-radius: 4px; font-size: 0.875rem;
                                        background-color: <?= $test['status'] === 'active' ? '#d1fae5' : '#fee2e2' ?>;
                                        color: <?= $test['status'] === 'active' ? '#065f46' : '#991b1b' ?>;">
                                        <?= e(ucfirst($test['status'])) ?>
                                    </span>
                                </td>
                                <?php if ($canManageLab): ?>
                                    <td>
                                        <div style="display: flex; gap: 0.5rem; flex-wrap: wrap;">
                                            <a href="?action=laboratory&sub=edit_test&id=<?= $test['id'] ?>" class="btn btn-secondary" style="padding: 0.5rem 1rem; font-size: 0.875rem;">Edit</a>
                                            <form method="POST" action="?action=laboratory" style="display: inline;">
                                                <input type="hidden" name="test_id" value="<?= $test['id'] ?>">
                                                <button type="submit" name="delete_test" class="btn btn-danger" style="padding: 0.5rem 1rem; font-size: 0.875rem;"
                                                        onclick="return confirm('Are you sure you want to delete this test?')">
                                                    Delete
                                                </button>
                                            </form>
                                        </div>
                                    </td>
                                <?php endif; ?>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <p style="text-align: center; padding: 3rem; color: var(--text-light);">
                No lab tests available yet. Click "Add New Test" to get started.
            </p>
        <?php endif; ?>
    </div>
<?php endif; ?>
