<?php
$db = Database::getInstance()->getConnection();

// Check if user has permission to write reports
$allowedRoles = ['nurse', 'doctor', 'pharmacist', 'pharmacy_technician', 'lab_scientist', 'lab_technician'];
if (!in_array($_SESSION['role'], $allowedRoles)) {
    echo '<div class="alert alert-error">Access denied. Only Nurses, Doctors, Pharmacists, Pharmacy Technicians, Lab Scientists, and Lab Technicians can write patient reports.</div>';
    return;
}

$tenantId = $_SESSION['tenant_id'];
$userId = $_SESSION['user_id'];
$userRole = $_SESSION['role'];
$userName = $_SESSION['full_name'];

// Handle form submission for new report
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_report'])) {
    $patientId = $_POST['patient_id'];
    $reportType = $_POST['report_type'];
    $reportTitle = $_POST['report_title'];
    $reportContent = $_POST['report_content'];

    // Generate report number
    $reportNumber = 'RPT' . date('Ymd') . str_pad($db->query("SELECT COUNT(*) FROM patient_reports WHERE tenant_id = $tenantId")->fetchColumn() + 1, 4, '0', STR_PAD_LEFT);

    // Insert report
    $stmt = $db->prepare("INSERT INTO patient_reports (tenant_id, patient_id, report_number, report_type, report_title, report_content, written_by, writer_role, writer_name)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");

    if ($stmt->execute([$tenantId, $patientId, $reportNumber, $reportType, $reportTitle, $reportContent, $userId, $userRole, $userName])) {
        $message = "Report created successfully! Report Number: $reportNumber";
    } else {
        $error = "Failed to create report.";
    }
}

// Get selected patient if any
$selectedPatient = null;
$myReports = [];
$otherReports = [];
if (isset($_GET['patient_id'])) {
    $patientId = $_GET['patient_id'];
    $stmt = $db->prepare("SELECT * FROM patients WHERE id = ? AND tenant_id = ?");
    $stmt->execute([$patientId, $tenantId]);
    $selectedPatient = $stmt->fetch(PDO::FETCH_ASSOC);

    // Get all reports for this patient - separated by ownership
    if ($selectedPatient) {
        // Get my reports (written by current user)
        $stmt = $db->prepare("SELECT * FROM patient_reports WHERE patient_id = ? AND tenant_id = ? AND written_by = ? ORDER BY created_at DESC");
        $stmt->execute([$patientId, $tenantId, $userId]);
        $myReports = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Get other team reports (written by other staff - view only)
        $stmt = $db->prepare("SELECT * FROM patient_reports WHERE patient_id = ? AND tenant_id = ? AND written_by != ? ORDER BY created_at DESC");
        $stmt->execute([$patientId, $tenantId, $userId]);
        $otherReports = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}

// Search patients
$searchResults = [];
if (isset($_GET['search']) && !empty($_GET['search_query'])) {
    $searchQuery = '%' . $_GET['search_query'] . '%';
    $stmt = $db->prepare("SELECT * FROM patients WHERE tenant_id = ? AND (full_name LIKE ? OR patient_number LIKE ?) LIMIT 10");
    $stmt->execute([$tenantId, $searchQuery, $searchQuery]);
    $searchResults = $stmt->fetchAll(PDO::FETCH_ASSOC);
}
?>

<div class="card">
    <h2>Patient Reports</h2>
    <p>Write comprehensive reports on patient care, observations, and activities</p>

    <?php if (isset($message)): ?>
        <div class="alert alert-success"><?= e($message) ?></div>
    <?php endif; ?>

    <?php if (isset($error)): ?>
        <div class="alert alert-error"><?= e($error) ?></div>
    <?php endif; ?>
</div>

<!-- Search Patient -->
<div class="card">
    <h3>Search Patient</h3>
    <form method="GET" action="?action=patient_reports">
        <input type="hidden" name="action" value="patient_reports">
        <div class="form-group">
            <label for="search_query">Search by Patient Name or Number</label>
            <div style="display: flex; gap: 0.5rem;">
                <input type="text" id="search_query" name="search_query" class="form-control"
                    placeholder="Enter patient name or number"
                    value="<?= e($_GET['search_query'] ?? '') ?>" required>
                <button type="submit" name="search" class="btn btn-primary">Search</button>
            </div>
        </div>
    </form>

    <?php if (!empty($searchResults)): ?>
        <div style="margin-top: 1rem;">
            <h4>Search Results:</h4>
            <table class="table">
                <thead>
                    <tr>
                        <th>Patient Number</th>
                        <th>Name</th>
                        <th>Gender</th>
                        <th>Age</th>
                        <th>Action</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($searchResults as $patient): ?>
                        <tr>
                            <td><?= e($patient['patient_number']) ?></td>
                            <td><?= e($patient['full_name']) ?></td>
                            <td><?= e($patient['gender']) ?></td>
                            <td><?php
                                $dob = new DateTime($patient['date_of_birth']);
                                $now = new DateTime();
                                echo $now->diff($dob)->y;
                            ?> years</td>
                            <td>
                                <a href="?action=patient_reports&patient_id=<?= $patient['id'] ?>" class="btn btn-primary">Select</a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>

<?php if ($selectedPatient): ?>
    <!-- Patient Information -->
    <div class="card" style="background: linear-gradient(135deg, #dbeafe 0%, #bfdbfe 100%); border-left: 4px solid #3b82f6;">
        <h3>Selected Patient</h3>
        <div class="grid">
            <div>
                <p><strong>Patient Number:</strong> <?= e($selectedPatient['patient_number']) ?></p>
                <p><strong>Full Name:</strong> <?= e($selectedPatient['full_name']) ?></p>
                <p><strong>Gender:</strong> <?= e($selectedPatient['gender']) ?></p>
            </div>
            <div>
                <p><strong>Age:</strong> <?php
                    $dob = new DateTime($selectedPatient['date_of_birth']);
                    $now = new DateTime();
                    echo $now->diff($dob)->y;
                ?> years</p>
                <p><strong>Phone:</strong> <?= e($selectedPatient['phone']) ?></p>
                <p><strong>Email:</strong> <?= e($selectedPatient['email']) ?></p>
            </div>
        </div>
    </div>

    <!-- Write New Report -->
    <div class="card">
        <h3>Write New Report</h3>
        <form method="POST" action="?action=patient_reports&patient_id=<?= $selectedPatient['id'] ?>">
            <input type="hidden" name="patient_id" value="<?= $selectedPatient['id'] ?>">

            <div class="form-group">
                <label for="report_type">Report Type</label>
                <select id="report_type" name="report_type" class="form-control" required>
                    <option value="">-- Select Report Type --</option>
                    <?php if ($userRole === 'nurse'): ?>
                        <option value="Nursing Assessment">Nursing Assessment</option>
                        <option value="Vital Signs Observation">Vital Signs Observation</option>
                        <option value="Nursing Care Plan">Nursing Care Plan</option>
                        <option value="Patient Progress">Patient Progress</option>
                        <option value="Wound Care">Wound Care</option>
                        <option value="Medication Administration">Medication Administration</option>
                    <?php endif; ?>

                    <?php if ($userRole === 'doctor'): ?>
                        <option value="Clinical Assessment">Clinical Assessment</option>
                        <option value="Diagnosis Report">Diagnosis Report</option>
                        <option value="Treatment Plan">Treatment Plan</option>
                        <option value="Follow-up Consultation">Follow-up Consultation</option>
                        <option value="Discharge Summary">Discharge Summary</option>
                        <option value="Specialist Review">Specialist Review</option>
                    <?php endif; ?>

                    <?php if ($userRole === 'pharmacist' || $userRole === 'pharmacy_technician'): ?>
                        <option value="Medication Dispensing">Medication Dispensing</option>
                        <option value="Drug Interaction Check">Drug Interaction Check</option>
                        <option value="Pharmaceutical Care">Pharmaceutical Care</option>
                        <option value="Patient Counseling">Patient Counseling</option>
                        <option value="Medication Review">Medication Review</option>
                    <?php endif; ?>

                    <?php if ($userRole === 'lab_scientist' || $userRole === 'lab_technician'): ?>
                        <option value="Laboratory Results">Laboratory Results</option>
                        <option value="Test Analysis">Test Analysis</option>
                        <option value="Specimen Collection">Specimen Collection</option>
                        <option value="Quality Control">Quality Control</option>
                        <option value="Test Interpretation">Test Interpretation</option>
                    <?php endif; ?>

                    <option value="General Observation">General Observation</option>
                    <option value="Other">Other</option>
                </select>
            </div>

            <div class="form-group">
                <label for="report_title">Report Title</label>
                <input type="text" id="report_title" name="report_title" class="form-control"
                    placeholder="Brief title for this report" required maxlength="200">
            </div>

            <div class="form-group">
                <label for="report_content">Report Content</label>
                <textarea id="report_content" name="report_content" class="form-control"
                    rows="12" placeholder="Write your detailed report here..." required></textarea>
                <small style="color: var(--text-light);">
                    Provide comprehensive details including observations, assessments, actions taken, and recommendations.
                </small>
            </div>

            <button type="submit" name="create_report" class="btn btn-primary btn-block">Submit Report</button>
        </form>
    </div>

    <!-- My Reports (Editable) -->
    <div class="card">
        <h3 style="color: #3b82f6;">📝 My Reports (<?= count($myReports) ?>)</h3>
        <p style="color: var(--text-light); font-size: 0.9rem; margin-bottom: 1.5rem;">
            Reports you have written - You can view and track your own contributions
        </p>

        <?php if (empty($myReports)): ?>
            <p style="color: var(--text-light); text-align: center; padding: 2rem; background: #f9fafb; border-radius: 8px;">
                You haven't written any reports for this patient yet. Use the form above to create your first report.
            </p>
        <?php else: ?>
            <?php foreach ($myReports as $report): ?>
                <div style="border: 1px solid var(--border-color); border-radius: 8px; padding: 1.5rem; margin-bottom: 1rem; background: <?php
                    // Color coding by role
                    $bgColor = '#f9fafb';
                    if ($report['writer_role'] === 'doctor') $bgColor = '#fef3c7';
                    if ($report['writer_role'] === 'nurse') $bgColor = '#dbeafe';
                    if ($report['writer_role'] === 'pharmacist' || $report['writer_role'] === 'pharmacy_technician') $bgColor = '#d1fae5';
                    if ($report['writer_role'] === 'lab_scientist' || $report['writer_role'] === 'lab_technician') $bgColor = '#fce7f3';
                    echo $bgColor;
                ?>;">
                    <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 1rem;">
                        <div>
                            <h4 style="margin-bottom: 0.5rem; color: var(--text-dark);"><?= e($report['report_title']) ?></h4>
                            <p style="margin: 0; font-size: 0.9rem; color: var(--text-light);">
                                <strong>Report #:</strong> <?= e($report['report_number']) ?> |
                                <strong>Type:</strong> <?= e($report['report_type']) ?>
                            </p>
                        </div>
                        <div style="text-align: right;">
                            <span style="display: inline-block; padding: 0.25rem 0.75rem; background: <?php
                                $roleColor = '#6b7280';
                                if ($report['writer_role'] === 'doctor') $roleColor = '#f59e0b';
                                if ($report['writer_role'] === 'nurse') $roleColor = '#3b82f6';
                                if ($report['writer_role'] === 'pharmacist' || $report['writer_role'] === 'pharmacy_technician') $roleColor = '#10b981';
                                if ($report['writer_role'] === 'lab_scientist' || $report['writer_role'] === 'lab_technician') $roleColor = '#ec4899';
                                echo $roleColor;
                            ?>; color: white; border-radius: 4px; font-size: 0.85rem; font-weight: 500;">
                                <?= e(ucwords(str_replace('_', ' ', $report['writer_role']))) ?>
                            </span>
                        </div>
                    </div>

                    <div style="background: white; padding: 1rem; border-radius: 6px; margin-bottom: 1rem;">
                        <p style="white-space: pre-wrap; margin: 0; line-height: 1.6;"><?= e($report['report_content']) ?></p>
                    </div>

                    <div style="display: flex; justify-content: space-between; font-size: 0.85rem; color: var(--text-light);">
                        <span><strong>Written by:</strong> <?= e($report['writer_name']) ?></span>
                        <span><strong>Date:</strong> <?= date('F j, Y g:i A', strtotime($report['created_at'])) ?></span>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>

    <!-- Team Reports (View Only) -->
    <div class="card" style="background: #f9fafb;">
        <h3 style="color: #6b7280;">👥 Team Reports - View Only (<?= count($otherReports) ?>)</h3>
        <p style="color: var(--text-light); font-size: 0.9rem; margin-bottom: 1.5rem;">
            Reports written by other healthcare team members - Read-only access for collaborative care
        </p>

        <?php if (empty($otherReports)): ?>
            <p style="color: var(--text-light); text-align: center; padding: 2rem; background: white; border-radius: 8px;">
                No reports from other team members yet.
            </p>
        <?php else: ?>
            <?php foreach ($otherReports as $report): ?>
                <div style="border: 2px solid #e5e7eb; border-radius: 8px; padding: 1.5rem; margin-bottom: 1rem; background: <?php
                    // Color coding by role
                    $bgColor = 'white';
                    if ($report['writer_role'] === 'doctor') $bgColor = '#fef3c7';
                    if ($report['writer_role'] === 'nurse') $bgColor = '#dbeafe';
                    if ($report['writer_role'] === 'pharmacist' || $report['writer_role'] === 'pharmacy_technician') $bgColor = '#d1fae5';
                    if ($report['writer_role'] === 'lab_scientist' || $report['writer_role'] === 'lab_technician') $bgColor = '#fce7f3';
                    echo $bgColor;
                ?>; position: relative;">
                    <!-- View Only Badge -->
                    <div style="position: absolute; top: 10px; right: 10px; background: #6b7280; color: white; padding: 0.25rem 0.75rem; border-radius: 4px; font-size: 0.75rem; font-weight: 600;">
                        VIEW ONLY
                    </div>

                    <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 1rem; padding-right: 100px;">
                        <div>
                            <h4 style="margin-bottom: 0.5rem; color: var(--text-dark);"><?= e($report['report_title']) ?></h4>
                            <p style="margin: 0; font-size: 0.9rem; color: var(--text-light);">
                                <strong>Report #:</strong> <?= e($report['report_number']) ?> |
                                <strong>Type:</strong> <?= e($report['report_type']) ?>
                            </p>
                        </div>
                    </div>

                    <div style="margin-bottom: 1rem;">
                        <span style="display: inline-block; padding: 0.25rem 0.75rem; background: <?php
                            $roleColor = '#6b7280';
                            if ($report['writer_role'] === 'doctor') $roleColor = '#f59e0b';
                            if ($report['writer_role'] === 'nurse') $roleColor = '#3b82f6';
                            if ($report['writer_role'] === 'pharmacist' || $report['writer_role'] === 'pharmacy_technician') $roleColor = '#10b981';
                            if ($report['writer_role'] === 'lab_scientist' || $report['writer_role'] === 'lab_technician') $roleColor = '#ec4899';
                            echo $roleColor;
                        ?>; color: white; border-radius: 4px; font-size: 0.85rem; font-weight: 500;">
                            <?= e(ucwords(str_replace('_', ' ', $report['writer_role']))) ?>
                        </span>
                    </div>

                    <div style="background: rgba(255, 255, 255, 0.7); padding: 1rem; border-radius: 6px; margin-bottom: 1rem; border-left: 4px solid #6b7280;">
                        <p style="white-space: pre-wrap; margin: 0; line-height: 1.6; color: #374151;"><?= e($report['report_content']) ?></p>
                    </div>

                    <div style="display: flex; justify-content: space-between; font-size: 0.85rem; color: var(--text-light);">
                        <span><strong>Written by:</strong> <?= e($report['writer_name']) ?> (<?= e(ucwords(str_replace('_', ' ', $report['writer_role']))) ?>)</span>
                        <span><strong>Date:</strong> <?= date('F j, Y g:i A', strtotime($report['created_at'])) ?></span>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php endif; ?>
    </div>
<?php else: ?>
    <div class="card" style="text-align: center; padding: 3rem;">
        <p style="color: var(--text-light); font-size: 1.1rem;">Search for a patient above to view and write reports</p>
    </div>
<?php endif; ?>
