<?php
$db = Database::getInstance()->getConnection();
$tenantId = $_SESSION['tenant_id'];
$userId = $_SESSION['user_id'];
$userRole = $_SESSION['role'];

// Check if user can access pharmacy prescriptions
$canAccessPharmacy = in_array($userRole, ['hospital_admin', 'pharmacist', 'pharmacy_technician']);

if (!$canAccessPharmacy) {
    header('Location: ?action=dashboard');
    exit;
}

$subAction = $_GET['sub'] ?? 'list';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['dispense_drugs'])) {
        // Generate invoice for dispensed drugs
        $prescriptionId = $_POST['prescription_id'];
        $totalAmount = 0;

        // Get prescription details
        $stmt = $db->prepare("
            SELECT pr.*, p.full_name as patient_name, p.patient_number
            FROM prescriptions pr
            JOIN patients p ON pr.patient_id = p.id
            WHERE pr.id = ? AND pr.tenant_id = ?
        ");
        $stmt->execute([$prescriptionId, $tenantId]);
        $prescription = $stmt->fetch(PDO::FETCH_ASSOC);

        // Generate invoice number
        $invoiceNumber = 'INV' . date('Ymd') . str_pad($db->query("SELECT COUNT(*) FROM billing WHERE tenant_id = $tenantId")->fetchColumn() + 1, 4, '0', STR_PAD_LEFT);

        // Get selected items to dispense
        $dispensedItems = [];
        $itemDescriptions = [];

        if (isset($_POST['items_to_dispense'])) {
            foreach ($_POST['items_to_dispense'] as $itemId) {
                // Get item details
                $itemStmt = $db->prepare("
                    SELECT pi.*, d.selling_price
                    FROM prescription_items pi
                    LEFT JOIN pharmacy_inventory d ON pi.drug_id = d.id
                    WHERE pi.id = ? AND pi.tenant_id = ?
                ");
                $itemStmt->execute([$itemId, $tenantId]);
                $item = $itemStmt->fetch(PDO::FETCH_ASSOC);

                if ($item && $item['selling_price']) {
                    $itemTotal = $item['selling_price'] * $item['quantity'];
                    $totalAmount += $itemTotal;
                    $dispensedItems[] = $item;
                    $itemDescriptions[] = "{$item['drug_name']} ({$item['quantity']}) - " . formatCurrency($itemTotal);

                    // Update inventory quantity
                    $updateStmt = $db->prepare("
                        UPDATE pharmacy_inventory
                        SET quantity = quantity - ?
                        WHERE id = ? AND tenant_id = ?
                    ");
                    $updateStmt->execute([$item['quantity'], $item['drug_id'], $tenantId]);
                }

                // Update item status
                $updateStmt = $db->prepare("
                    UPDATE prescription_items
                    SET status = 'dispensed'
                    WHERE id = ?
                ");
                $updateStmt->execute([$itemId]);
            }
        }

        // Create billing invoice
        if ($totalAmount > 0) {
            $description = "Pharmacy - Prescription #{$prescription['prescription_number']}: " . implode(', ', $itemDescriptions);

            $stmt = $db->prepare("
                INSERT INTO billing (tenant_id, patient_id, invoice_number, description, amount, payment_status)
                VALUES (?, ?, ?, ?, ?, 'pending')
            ");
            $stmt->execute([
                $tenantId,
                $prescription['patient_id'],
                $invoiceNumber,
                $description,
                $totalAmount
            ]);

            $invoiceId = $db->lastInsertId();

            // Update items with invoice ID
            foreach ($_POST['items_to_dispense'] as $itemId) {
                $updateStmt = $db->prepare("
                    UPDATE prescription_items
                    SET invoice_id = ?
                    WHERE id = ?
                ");
                $updateStmt->execute([$invoiceId, $itemId]);
            }

            // Update prescription status
            $stmt = $db->prepare("
                UPDATE prescriptions
                SET status = 'invoiced'
                WHERE id = ?
            ");
            $stmt->execute([$prescriptionId]);

            $message = "Prescription dispensed! Invoice {$invoiceNumber} created for " . formatCurrency($totalAmount);
        } else {
            $error = "No items were selected for dispensing or no pricing available.";
        }

        $subAction = 'list';
    }
}

// Get all pending prescriptions
$pendingPrescriptions = $db->query("
    SELECT pr.*, p.full_name as patient_name, p.patient_number,
           u.full_name as doctor_name,
           COUNT(pi.id) as item_count
    FROM prescriptions pr
    JOIN patients p ON pr.patient_id = p.id
    LEFT JOIN users u ON pr.doctor_id = u.id
    LEFT JOIN prescription_items pi ON pr.id = pi.prescription_id
    WHERE pr.tenant_id = $tenantId AND pr.status = 'pending'
    GROUP BY pr.id
    ORDER BY pr.created_at DESC
")->fetchAll(PDO::FETCH_ASSOC);

// Get prescription details if viewing
$viewPrescription = null;
$prescriptionItems = [];

if ($subAction === 'view' && isset($_GET['id'])) {
    $stmt = $db->prepare("
        SELECT pr.*, p.full_name as patient_name, p.patient_number, p.phone,
               u.full_name as doctor_name
        FROM prescriptions pr
        JOIN patients p ON pr.patient_id = p.id
        LEFT JOIN users u ON pr.doctor_id = u.id
        WHERE pr.id = ? AND pr.tenant_id = ?
    ");
    $stmt->execute([$_GET['id'], $tenantId]);
    $viewPrescription = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($viewPrescription) {
        $stmt = $db->prepare("
            SELECT pi.*, d.quantity as stock_available, d.selling_price
            FROM prescription_items pi
            LEFT JOIN pharmacy_inventory d ON pi.drug_id = d.id
            WHERE pi.prescription_id = ? AND pi.tenant_id = ?
            ORDER BY pi.id
        ");
        $stmt->execute([$_GET['id'], $tenantId]);
        $prescriptionItems = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
?>

<div class="hero">
    <h1>Pharmacy Prescriptions</h1>
    <p>View and dispense drug prescriptions</p>
</div>

<?php if (isset($message)): ?>
    <div class="alert alert-success"><?= e($message) ?></div>
<?php endif; ?>

<?php if (isset($error)): ?>
    <div class="alert alert-error"><?= e($error) ?></div>
<?php endif; ?>

<?php if ($subAction === 'view' && $viewPrescription): ?>
    <!-- Prescription Details -->
    <div class="card">
        <div style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 1rem; margin-bottom: 1.5rem;">
            <h2>Prescription #<?= e($viewPrescription['prescription_number']) ?></h2>
            <a href="?action=pharmacy_prescriptions" class="btn btn-secondary">← Back to List</a>
        </div>

        <!-- Patient & Doctor Info -->
        <div style="background: var(--light-color); padding: 1.5rem; border-radius: 8px; margin-bottom: 2rem;">
            <div class="grid">
                <div>
                    <p><strong>Patient:</strong> <?= e($viewPrescription['patient_name']) ?> (<?= e($viewPrescription['patient_number']) ?>)</p>
                    <p><strong>Phone:</strong> <?= e($viewPrescription['phone'] ?? 'N/A') ?></p>
                </div>
                <div>
                    <p><strong>Prescribing Doctor:</strong> <?= e($viewPrescription['doctor_name']) ?></p>
                    <p><strong>Date:</strong> <?= e(date('M j, Y g:i A', strtotime($viewPrescription['created_at']))) ?></p>
                </div>
            </div>
        </div>

        <!-- Prescription Items -->
        <form method="POST" action="?action=pharmacy_prescription">
            <input type="hidden" name="prescription_id" value="<?= $viewPrescription['id'] ?>">

            <h3>Drugs to Dispense</h3>

            <?php if (count($prescriptionItems) > 0): ?>
                <div style="overflow-x: auto; margin-bottom: 1.5rem;">
                    <table class="table">
                        <thead>
                            <tr>
                                <th style="width: 40px;">Dispense</th>
                                <th>Drug</th>
                                <th>Dosage</th>
                                <th>Frequency</th>
                                <th>Duration</th>
                                <th>Quantity</th>
                                <th>Instructions</th>
                                <th>Stock</th>
                                <th>Unit Price</th>
                                <th>Total</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $prescriptionTotal = 0;
                            foreach ($prescriptionItems as $item):
                                $canDispense = $item['stock_available'] >= $item['quantity'];
                                $itemPrice = $item['selling_price'] ?? 0;
                                $itemTotal = $itemPrice * $item['quantity'];
                                $prescriptionTotal += $itemTotal;
                            ?>
                                <tr style="<?= !$canDispense ? 'background-color: #fee2e2;' : '' ?>">
                                    <td>
                                        <?php if ($item['status'] === 'pending'): ?>
                                            <input type="checkbox" name="items_to_dispense[]" value="<?= $item['id'] ?>"
                                                   <?= !$canDispense ? 'disabled' : '' ?>>
                                        <?php else: ?>
                                            <span style="color: green;">✓</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><strong><?= e($item['drug_name']) ?></strong></td>
                                    <td><?= e($item['dosage']) ?></td>
                                    <td><?= e($item['frequency']) ?></td>
                                    <td><?= e($item['duration']) ?></td>
                                    <td><?= e($item['quantity']) ?></td>
                                    <td><?= e($item['instructions'] ?? '-') ?></td>
                                    <td>
                                        <span style="<?= $canDispense ? 'color: green;' : 'color: red;' ?>">
                                            <?= $item['stock_available'] ?? 'N/A' ?>
                                        </span>
                                    </td>
                                    <td><?= formatCurrency($itemPrice) ?></td>
                                    <td><strong><?= formatCurrency($itemTotal) ?></strong></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                        <tfoot>
                            <tr style="background-color: var(--primary-color); color: white;">
                                <td colspan="9" style="text-align: right; padding: 1rem;"><strong>Total:</strong></td>
                                <td style="padding: 1rem;"><strong><?= formatCurrency($prescriptionTotal) ?></strong></td>
                            </tr>
                        </tfoot>
                    </table>
                </div>

                <div class="alert alert-info" style="margin-bottom: 1.5rem;">
                    <strong>Note:</strong> Select drugs that are in stock and click "Dispense & Generate Invoice". The patient will take the invoice to accounting for payment.
                </div>

                <?php $hasPendingItems = false; ?>
                <?php foreach ($prescriptionItems as $item): ?>
                    <?php if ($item['status'] === 'pending' && $item['stock_available'] >= $item['quantity']): ?>
                        <?php $hasPendingItems = true; break; ?>
                    <?php endif; ?>
                <?php endforeach; ?>

                <?php if ($hasPendingItems): ?>
                    <button type="submit" name="dispense_drugs" class="btn btn-primary">Dispense & Generate Invoice</button>
                <?php endif; ?>
            <?php else: ?>
                <p style="text-align: center; padding: 2rem; color: var(--text-light);">
                    No drugs found in this prescription.
                </p>
            <?php endif; ?>

            <a href="?action=pharmacy_prescriptions" class="btn btn-secondary">Back to List</a>
        </form>
    </div>

<?php else: ?>
    <!-- Pending Prescriptions List -->
    <div class="card">
        <h2>Pending Prescriptions (<?= count($pendingPrescriptions) ?>)</h2>

        <?php if (count($pendingPrescriptions) > 0): ?>
            <div style="overflow-x: auto;">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Prescription #</th>
                            <th>Patient</th>
                            <th>Patient Number</th>
                            <th>Doctor</th>
                            <th>Items</th>
                            <th>Date</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($pendingPrescriptions as $prescription): ?>
                            <tr>
                                <td><strong><?= e($prescription['prescription_number']) ?></strong></td>
                                <td><?= e($prescription['patient_name']) ?></td>
                                <td><?= e($prescription['patient_number']) ?></td>
                                <td><?= e($prescription['doctor_name']) ?></td>
                                <td><?= $prescription['item_count'] ?></td>
                                <td><?= e(date('M j, Y g:i A', strtotime($prescription['created_at']))) ?></td>
                                <td>
                                    <a href="?action=pharmacy_prescriptions&sub=view&id=<?= $prescription['id'] ?>" class="btn btn-primary" style="padding: 0.5rem 1rem; font-size: 0.875rem;">
                                        View & Dispense
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <p style="text-align: center; padding: 3rem; color: var(--text-light);">
                No pending prescriptions to dispense.
            </p>
        <?php endif; ?>
    </div>

    <!-- All Prescriptions History -->
    <div class="card">
        <h2>Prescription History</h2>

        <?php
        $allPrescriptions = $db->query("
            SELECT pr.*, p.full_name as patient_name, p.patient_number
            FROM prescriptions pr
            JOIN patients p ON pr.patient_id = p.id
            WHERE pr.tenant_id = $tenantId
            ORDER BY pr.created_at DESC
            LIMIT 50
        ")->fetchAll(PDO::FETCH_ASSOC);
        ?>

        <?php if (count($allPrescriptions) > 0): ?>
            <div style="overflow-x: auto;">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Prescription #</th>
                            <th>Patient</th>
                            <th>Status</th>
                            <th>Date</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($allPrescriptions as $prescription): ?>
                            <tr>
                                <td><?= e($prescription['prescription_number']) ?></td>
                                <td><?= e($prescription['patient_name']) ?></td>
                                <td>
                                    <span style="padding: 0.25rem 0.75rem; border-radius: 4px; font-size: 0.875rem;
                                        background-color: <?= $prescription['status'] === 'pending' ? '#fef3c7' : '#d1fae5' ?>;
                                        color: <?= $prescription['status'] === 'pending' ? '#92400e' : '#065f46' ?>;">
                                        <?= ucfirst($prescription['status']) ?>
                                    </span>
                                </td>
                                <td><?= e(date('M j, Y', strtotime($prescription['created_at']))) ?></td>
                                <td>
                                    <a href="?action=pharmacy_prescriptions&sub=view&id=<?= $prescription['id'] ?>" class="btn btn-secondary" style="padding: 0.5rem 1rem; font-size: 0.875rem;">
                                        View
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
<?php endif; ?>
