<?php
$auth = new Auth();

// Check if user can manage users
if (!$auth->canManageUsers()) {
    header('Location: ?action=dashboard');
    exit;
}

$db = Database::getInstance()->getConnection();
$tenantId = $_SESSION['tenant_id'];
$subAction = $_GET['sub'] ?? 'list';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_user'])) {
        // Check if username exists
        $stmt = $db->prepare("SELECT COUNT(*) FROM users WHERE username = ?");
        $stmt->execute([$_POST['username']]);

        if ($stmt->fetchColumn() > 0) {
            $error = 'Username already exists. Please choose another one.';
        } else {
            // Generate employee number
            $empNumber = 'EMP' . str_pad($db->query("SELECT COUNT(*) FROM users WHERE tenant_id = $tenantId")->fetchColumn() + 1, 4, '0', STR_PAD_LEFT);

            $hashedPassword = password_hash($_POST['password'], PASSWORD_DEFAULT);

            $stmt = $db->prepare("
                INSERT INTO users (tenant_id, username, password, full_name, email, phone, role, department, employee_number, created_by)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");

            $stmt->execute([
                $tenantId,
                $_POST['username'],
                $hashedPassword,
                $_POST['full_name'],
                $_POST['email'],
                $_POST['phone'],
                $_POST['role'],
                $_POST['department'],
                $empNumber,
                $_SESSION['user_id']
            ]);

            // If role is doctor, create doctor record
            if ($_POST['role'] === 'doctor') {
                $userId = $db->lastInsertId();
                $docNumber = 'DOC' . str_pad($db->query("SELECT COUNT(*) FROM doctors WHERE tenant_id = $tenantId")->fetchColumn() + 1, 3, '0', STR_PAD_LEFT);

                $stmt = $db->prepare("
                    INSERT INTO doctors (tenant_id, user_id, doctor_number, full_name, specialization, phone, email, consultation_fee)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                ");

                $stmt->execute([
                    $tenantId,
                    $userId,
                    $docNumber,
                    $_POST['full_name'],
                    $_POST['department'],
                    $_POST['phone'],
                    $_POST['email'],
                    $_POST['consultation_fee'] ?? 0
                ]);
            }

            $message = 'User created successfully! Employee Number: ' . $empNumber;
            $subAction = 'list';
        }
    } elseif (isset($_POST['update_user'])) {
        $stmt = $db->prepare("
            UPDATE users SET
                full_name = ?, email = ?, phone = ?, role = ?, department = ?, status = ?
            WHERE id = ? AND tenant_id = ?
        ");

        $stmt->execute([
            $_POST['full_name'],
            $_POST['email'],
            $_POST['phone'],
            $_POST['role'],
            $_POST['department'],
            $_POST['status'],
            $_POST['user_id'],
            $tenantId
        ]);

        $message = 'User updated successfully!';
        $subAction = 'list';
    } elseif (isset($_POST['reset_password'])) {
        $newPassword = bin2hex(random_bytes(8));
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);

        $stmt = $db->prepare("UPDATE users SET password = ? WHERE id = ? AND tenant_id = ?");
        $stmt->execute([$hashedPassword, $_POST['user_id'], $tenantId]);

        $message = "Password reset successfully! New password: {$newPassword} (Please note this down)";
    }
}

// Get user for editing
$editUser = null;
if ($subAction === 'edit' && isset($_GET['id'])) {
    $stmt = $db->prepare("SELECT * FROM users WHERE id = ? AND tenant_id = ?");
    $stmt->execute([$_GET['id'], $tenantId]);
    $editUser = $stmt->fetch(PDO::FETCH_ASSOC);
}

// Get all users
$users = $db->query("SELECT * FROM users WHERE tenant_id = $tenantId ORDER BY created_at DESC")->fetchAll(PDO::FETCH_ASSOC);

// Count users by role
$usersByRole = [];
foreach (ROLES as $roleKey => $roleName) {
    if ($roleKey !== 'super_admin') {
        $count = $db->query("SELECT COUNT(*) FROM users WHERE tenant_id = $tenantId AND role = '$roleKey'")->fetchColumn();
        if ($count > 0) {
            $usersByRole[$roleKey] = ['name' => $roleName, 'count' => $count];
        }
    }
}
?>

<div class="hero">
    <h1>User Management</h1>
    <p>Create and manage users for your hospital</p>
</div>

<?php if (isset($message)): ?>
    <div class="alert alert-success"><?= e($message) ?></div>
<?php endif; ?>

<?php if (isset($error)): ?>
    <div class="alert alert-error"><?= e($error) ?></div>
<?php endif; ?>

<!-- User Statistics -->
<div class="card">
    <h2>Users by Role</h2>
    <div class="stats">
        <?php foreach ($usersByRole as $roleKey => $roleInfo): ?>
            <div class="stat-card">
                <div class="stat-value"><?= $roleInfo['count'] ?></div>
                <div class="stat-label"><?= e($roleInfo['name']) ?></div>
            </div>
        <?php endforeach; ?>
    </div>
</div>

<?php if ($subAction === 'add' || $subAction === 'edit'): ?>
    <!-- Add/Edit User Form -->
    <div class="card">
        <h2><?= $subAction === 'edit' ? 'Edit User' : 'Create New User' ?></h2>
        <form method="POST" action="?action=users">
            <?php if ($subAction === 'edit'): ?>
                <input type="hidden" name="user_id" value="<?= e($editUser['id']) ?>">
            <?php endif; ?>

            <div class="grid">
                <div class="form-group">
                    <label for="username">Username * <?= $subAction === 'edit' ? '(Cannot be changed)' : '' ?></label>
                    <input type="text" id="username" name="username" class="form-control"
                           value="<?= e($editUser['username'] ?? '') ?>"
                           <?= $subAction === 'edit' ? 'readonly' : 'required' ?>>
                </div>

                <?php if ($subAction === 'add'): ?>
                    <div class="form-group">
                        <label for="password">Password *</label>
                        <input type="password" id="password" name="password" class="form-control" required minlength="6">
                        <small style="color: var(--text-light);">Minimum 6 characters</small>
                    </div>
                <?php endif; ?>

                <div class="form-group">
                    <label for="full_name">Full Name *</label>
                    <input type="text" id="full_name" name="full_name" class="form-control"
                           value="<?= e($editUser['full_name'] ?? '') ?>" required>
                </div>

                <div class="form-group">
                    <label for="email">Email</label>
                    <input type="email" id="email" name="email" class="form-control"
                           value="<?= e($editUser['email'] ?? '') ?>">
                </div>

                <div class="form-group">
                    <label for="phone">Phone</label>
                    <input type="tel" id="phone" name="phone" class="form-control"
                           value="<?= e($editUser['phone'] ?? '') ?>">
                </div>

                <div class="form-group">
                    <label for="role">Role *</label>
                    <select id="role" name="role" class="form-control" required onchange="toggleDoctorFields(this.value)">
                        <option value="">Select Role</option>
                        <?php foreach (ROLES as $roleKey => $roleName): ?>
                            <?php if ($roleKey !== 'super_admin' && $roleKey !== 'hospital_admin'): ?>
                                <option value="<?= $roleKey ?>" <?= ($editUser['role'] ?? '') === $roleKey ? 'selected' : '' ?>>
                                    <?= e($roleName) ?>
                                </option>
                            <?php endif; ?>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label for="department">Department/Specialization</label>
                    <input type="text" id="department" name="department" class="form-control"
                           value="<?= e($editUser['department'] ?? '') ?>"
                           placeholder="e.g., Cardiology, Emergency, Lab">
                </div>

                <?php if ($subAction === 'edit'): ?>
                    <div class="form-group">
                        <label for="status">Status</label>
                        <select id="status" name="status" class="form-control">
                            <option value="active" <?= ($editUser['status'] ?? '') === 'active' ? 'selected' : '' ?>>Active</option>
                            <option value="inactive" <?= ($editUser['status'] ?? '') === 'inactive' ? 'selected' : '' ?>>Inactive</option>
                        </select>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Doctor specific fields -->
            <div id="doctor_fields" style="display: none;">
                <h3 style="margin-top: 1.5rem; margin-bottom: 1rem;">Doctor Information</h3>
                <div class="form-group">
                    <label for="consultation_fee">Consultation Fee ($)</label>
                    <input type="number" id="consultation_fee" name="consultation_fee" class="form-control"
                           step="0.01" min="0" value="0">
                </div>
            </div>

            <div style="display: flex; gap: 1rem; margin-top: 1.5rem;">
                <button type="submit" name="<?= $subAction === 'edit' ? 'update_user' : 'add_user' ?>" class="btn btn-primary">
                    <?= $subAction === 'edit' ? 'Update User' : 'Create User' ?>
                </button>
                <a href="?action=users" class="btn btn-secondary">Cancel</a>
            </div>
        </form>
    </div>

    <script>
        function toggleDoctorFields(role) {
            const doctorFields = document.getElementById('doctor_fields');
            if (role === 'doctor') {
                doctorFields.style.display = 'block';
            } else {
                doctorFields.style.display = 'none';
            }
        }

        // Initialize on page load
        <?php if ($subAction === 'add' || ($subAction === 'edit' && $editUser['role'] === 'doctor')): ?>
            toggleDoctorFields('<?= $editUser['role'] ?? '' ?>');
        <?php endif; ?>
    </script>
<?php else: ?>
    <!-- User List -->
    <div class="card">
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1.5rem; flex-wrap: wrap; gap: 1rem;">
            <h2>All Users (<?= count($users) ?>)</h2>
            <a href="?action=users&sub=add" class="btn btn-primary">+ Add New User</a>
        </div>

        <?php if (count($users) > 0): ?>
            <div style="overflow-x: auto;">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Employee #</th>
                            <th>Username</th>
                            <th>Full Name</th>
                            <th>Role</th>
                            <th>Department</th>
                            <th>Contact</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): ?>
                            <tr>
                                <td><strong><?= e($user['employee_number'] ?? 'N/A') ?></strong></td>
                                <td><?= e($user['username']) ?></td>
                                <td><?= e($user['full_name']) ?></td>
                                <td><?= e(getRoleName($user['role'])) ?></td>
                                <td><?= e($user['department'] ?? 'N/A') ?></td>
                                <td>
                                    <?php if ($user['phone']): ?>
                                        <div><?= e($user['phone']) ?></div>
                                    <?php endif; ?>
                                    <?php if ($user['email']): ?>
                                        <small style="color: var(--text-light);"><?= e($user['email']) ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span style="padding: 0.25rem 0.75rem; border-radius: 4px; font-size: 0.875rem;
                                        background-color: <?= $user['status'] === 'active' ? '#d1fae5' : '#fee2e2' ?>;
                                        color: <?= $user['status'] === 'active' ? '#065f46' : '#991b1b' ?>;">
                                        <?= e(ucfirst($user['status'])) ?>
                                    </span>
                                </td>
                                <td>
                                    <div style="display: flex; gap: 0.5rem; flex-wrap: wrap;">
                                        <a href="?action=users&sub=edit&id=<?= $user['id'] ?>" class="btn btn-secondary" style="padding: 0.5rem 1rem; font-size: 0.875rem;">Edit</a>
                                        <form method="POST" action="?action=users" style="display: inline;">
                                            <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                            <button type="submit" name="reset_password" class="btn btn-secondary" style="padding: 0.5rem 1rem; font-size: 0.875rem;"
                                                    onclick="return confirm('Reset password for this user?')">
                                                Reset Password
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <p style="text-align: center; padding: 3rem; color: var(--text-light);">
                No users created yet. Click "Add New User" to get started.
            </p>
        <?php endif; ?>
    </div>
<?php endif; ?>
